/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

#include <builtin.h>
#include <math.h>
#include <stdio.h>
#include <stdarg.h>
#include <values.h>
#include <Obstack.h>
#include <AllocQueue.h>
#include <float.h>

/*
 common functions on built-in types
*/

long gcd(long x, long y)        // euclid's algorithm
{
  long a = abs(x);
  long b = abs(y);

  long tmp;
  
  if (b > a)
  {
    tmp = a; a = b; b = tmp;
  }
  for(;;)
  {
    if (b == 0)
      return a;
    else if (b == 1)
      return b;
    else
    {
      tmp = b;
      b = a % b;
      a = tmp;
    }
  }
}

double pow(double x, long p)
{
  if (p == 0)
    return (x < 0 && (p & 1)) ? -1.0 : 1.0;
  else if (x == 0.0)
    return 0.0;
  else
  {
    double b;
    if (p < 0)
    {
      p = -p;
      b = 1.0 / x;
    }
    else
      b = x;
    double r = 1.0;
    for(;;)
    {
      if (p & 1)
        r *= b;
      if ((p >>= 1) == 0)
        return r;
      else
        b *= b;
    }
  }
}

long lg(long x)
{
  long l = 0;
  unsigned long a = abs(x);

  while (a > 1)
  {
    a = a >> 1;
    ++l;
  }
  return l;
}

long  pow(long  x, long y)
{
  if (x == 0)
    return (y == 0)? 1 : 0;
  else if (y < 0)
    return 0;
  else if (y == 0 || x == 1)
    return  1;
  else if (x == -1)
    return (y & 1)? -1 : 1;
  else
  {
    long r = 1;
    for(;;)
    {
      if (y & 1)
        r *= x;
      if ((y >>= 1) == 0)
        return r;
      else
        x *= x;
    }
  }
}

long sqrt(long x)
{
  if (x <= 0)
    return 0;                   // no int error handler, so ...
  else if (x == 1)
    return 1;
  else
  {
    long r = x >> 1;
    long q;
    for(;;)
    {
      q = x / r;
      if (q >= r)
        return r;
      else
        r = (r + q) >> 1;
    }
  }
}


// Obstacks are used as an easy way to allocate enough space
// for various input operations


Obstack _libgxx_io_ob;
char* _libgxx_io_oblast = 0;

    
// AllocQueues are used for output operations
// We guaranteee that the last _libgxx_maxfmt formats
// will be intact

static const int _libgxx_maxfmt = 20;
AllocQueue _libgxx_fmtq(_libgxx_maxfmt);

char* form(const char* fmt ...)
{
  va_list args;
  va_start(args, fmt);
  char* fmtbase = _libgxx_fmtq.alloc(BUFSIZ);
#ifndef HAVE_VPRINTF
  FILE b;
  b._flag = _IOWRT|_IOSTRG;
  b._ptr = fmtbase;
  b._cnt = BUFSIZ-1;
  _doprnt(fmt, args, &b);
  putc('\0', &b);
#else
  vsprintf(fmtbase, fmt, args);
#endif
  va_end(args);
  return fmtbase;
}

char* itoa(long x, int base = 10, int width = 0)
{
  int wrksiz;
  if (base == 10)
    wrksiz = width + 13;
  else
    wrksiz = (BITS(long) + 1) / lg(base) + 1 + width + 1;

  char* fmtbase = _libgxx_fmtq.alloc(wrksiz);
  char* e = fmtbase + wrksiz - 1;
  char* s = e;
  *--s = 0;
  char sgn = 0;

  if (x == 0)
    *--s = '0';
  else
  {
    int z;
    if (x < 0)
    {
      sgn = '-';
      z = -x;
    }
    else
      z = x;
    while (z != 0)
    {
      char ch = char(z % base);
      z = z / base;
      if (ch >= 10)
        ch += 'a' - 10;
      else
        ch += '0';
      *--s = ch;
    }
  }

  if (sgn) *--s = sgn;
  int w = e - s - 1;
  while (w++ < width)
    *--s = ' ';
  return s;
}

char* itoa(unsigned long x,  int base = 10, int width = 0)
{
  int wrksiz;
  if (base == 10)
    wrksiz = width + 13;
  else
    wrksiz = (BITS(long) + 1) / lg(base) + 1 + width + 1;

  char* fmtbase = _libgxx_fmtq.alloc(wrksiz);
  char* e = fmtbase + wrksiz - 1;
  char* s = e;
  *--s = 0;

  if (x == 0)
    *--s = '0';
  else
  {
    unsigned int b = base;
    while (x != 0)
    {
      char ch = char(x % b);
      x = x / b;
      if (ch >= 10)
        ch += 'a' - 10;
      else
        ch += '0';
      *--s = ch;
    }
  }

  int w = e - s - 1;
  while (w++ < width)
    *--s = ' ';
  return s;
}

char* itoa(long long x, int base = 10, int width = 0)
{
  int wrksiz;
  if (base == 10)
    wrksiz = width + 23;
  else
    wrksiz = (BITS(long long) + 1) / lg(base) + 1 + width + 1;

  char* fmtbase = _libgxx_fmtq.alloc(wrksiz);
  char* e = fmtbase + wrksiz - 1;
  char* s = e;
  *--s = 0;
  char sgn = 0;

  if (x == 0)
    *--s = '0';
  else
  {
    long long z;
    if (x < 0)
    {
      sgn = '-';
      z = -x;
    }
    else
      z = x;
    while (z != 0)
    {
      char ch = char(z % base);
      z = z / base;
      if (ch >= 10)
        ch += 'a' - 10;
      else
        ch += '0';
      *--s = ch;
    }
  }

  if (sgn) *--s = sgn;
  int w = e - s - 1;
  while (w++ < width)
    *--s = ' ';
  return s;
}

char* itoa(unsigned long long x,  int base = 10, int width = 0)
{
  int wrksiz;
  if (base == 10)
    wrksiz = width + 23;
  else
    wrksiz = (BITS(long long) + 1) / lg(base) + 1 + width + 1;

  char* fmtbase = _libgxx_fmtq.alloc(wrksiz);
  char* e = fmtbase + wrksiz - 1;
  char* s = e;
  *--s = 0;

  if (x == 0)
    *--s = '0';
  else
  {
    unsigned int b = base;
    while (x != 0)
    {
      char ch = char(x % b);
      x = x / b;
      if (ch >= 10)
        ch += 'a' - 10;
      else
        ch += '0';
      *--s = ch;
    }
  }

  int w = e - s - 1;
  while (w++ < width)
    *--s = ' ';
  return s;
}

char* hex(long i, int width = 0)
{
  return itoa(i, 16, width);
}

char* oct(long i, int width = 0)
{
  return itoa(i, 8, width);
}

char* dec(long i, int width = 0)
{
  return itoa(i, 10, width);
}

char* hex(unsigned long i, int width = 0)
{
  return itoa(i, 16, width);
}

char* oct(unsigned long i, int width = 0)
{
  return itoa(i, 8, width);
}

char* dec(unsigned long i, int width = 0)
{
  return itoa(i, 10, width);
}


char* dtoa(double fpnum,  char cvt = 'g', int width = 0, int prec = 6)
{
  // set up workspace

  const int worksiz = DBL_DIG + DBL_MAX_10_EXP + 2; // max possible digits

  // for fractional part
  char  fwork[worksiz];
  char* fworkend = &fwork[sizeof(fwork) - 1];
  char* fw = fwork;

  // for integer part
  char  iwork[worksiz];
  char* iworkend = &iwork[sizeof(iwork) - 1];
  char* iw = iworkend;
  *iw = 0;

  // for exponent part
  const int eworksiz = 30;
  char  ework[eworksiz];
  char* eworkend = &ework[sizeof(ework) - 1];
  char* ew = eworkend;
  *ew = 0;

  // grab sign & make non-negative
  int is_neg = fpnum < 0;
  if (is_neg) fpnum = -fpnum;

  // precision matters

  if (prec > worksiz - 2) // can't have more prec than supported
    prec = worksiz - 2;
  
  double powprec;
  if (prec == 6)
    powprec = 1.0e6;
  else
    powprec = pow(10.0, prec);

  double rounder = 0.5 / powprec;

  int f_fmt = cvt == 'f' ||
    ((cvt == 'g') && (fpnum == 0.0 || (fpnum >= 1e-4 && fpnum < powprec)));

  int iwidth = 0;
  int fwidth = 0;
  int ewidth = 0;

  if (f_fmt)  // fixed format
  {
    double ipart;
    double fpart = modf(fpnum, &ipart);

    // convert fractional part

    if (fpart >= rounder || cvt != 'g')
    {
      fpart += rounder;
      if (fpart >= 1.0)
      {
        ipart += 1.0;
        fpart -= 1.0;
      }
      double ffpart = fpart;
      double ifpart;
      for (int i = 0; i < prec; ++i)
      {
        ffpart = modf(ffpart * 10.0, &ifpart);
        *fw++ = '0' + int(ifpart);
        ++fwidth;
      }
      if (cvt == 'g')  // inhibit trailing zeroes if g-fmt
      {
        for (char* p = fw - 1; p >= fwork && *p == '0'; --p)
        {
          *p = 0;
          --fwidth;
        }
      }
    }

    // convert integer part
    if (ipart == 0.0)
    {
      if (cvt != 'g' || fwidth < prec || fwidth < width)
      {
        *--iw = '0'; ++iwidth;
      }
    }
    else if (ipart <= double(MAXLONG)) // a useful speedup
    {
      long li = long(ipart);
      while (li != 0)
      {
        *--iw = '0' + (li % 10);
        li = li / 10;
        ++iwidth;
      }
    }
    else // the slow way
    {
      while (ipart > 0.5)
      {
        double ff = modf(ipart / 10.0, &ipart);
        ff = (ff + 0.05) * 10.0;
        *--iw = '0' + int(ff);
        ++iwidth;
      }
    }

    // g-fmt: kill part of frac if prec/width exceeded
    if (cvt == 'g')
    {
      int m = prec;
      if (m < width)
        m = width;
      int adj = iwidth + fwidth - m;
      if (adj > fwidth)
        adj = fwidth;
      if (adj > 0)
      {
        int carry = 0;
        for (char* f = &fwork[fwidth-1]; f >= fwork && adj > 0; --adj, --f)
        {
          --fwidth;
          char ch = *f;
          *f = 0;
          if (ch > '5') // properly round: unavoidable propagation
          {
            int carry = 1;
            for (char* p = f - 1; p >= fwork && carry; --p)
            {
              ++*p;
              if (*p > '9')
                *p = '0';
              else
                carry = 0;
            }
            if (carry)
            {
              for (p = iworkend - 1; p >= iw && carry; --p)
              {
                ++*p;
                if (*p > '9')
                  *p = '0';
                else
                  carry = 0;
              }
              if (carry)
              {
                *--iw = '1';
                ++iwidth;
                --adj;
              }
            }
          }
        }
      }
    }
              
  }
  else  // e-fmt
  {
    
    // normalize
    int exp = 0;
    while (fpnum >= 10.0)
    {
      fpnum *= 0.1;
      ++exp;
    }
    while (fpnum > 0.0 && fpnum < 1.0)
    {
      fpnum *= 10.0;
      --exp;
    }
    
    double ipart;
    double fpart = modf(fpnum, &ipart);


    if (cvt == 'g')     // used up one digit for int part...
    {
      --prec;
      powprec /= 10.0;
      rounder = 0.5 / powprec;
    }

    // convert fractional part -- almost same as above
    if (fpart >= rounder || cvt != 'g')
    {
      fpart += rounder;
      if (fpart >= 1.0)
      {
        fpart -= 1.0;
        ipart += 1.0;
        if (ipart >= 10.0)
        {
          ++exp;
          ipart /= 10.0;
          fpart /= 10.0;
        }
      }
      double ffpart = fpart;
      double ifpart;
      for (int i = 0; i < prec; ++i)
      {
        ffpart = modf(ffpart * 10.0, &ifpart);
        *fw++ = '0' + int(ifpart);
        ++fwidth;
      }
      if (cvt == 'g')  // inhibit trailing zeroes if g-fmt
      {
        for (char* p = fw - 1; p >= fwork && *p == '0'; --p)
        {
          *p = 0;
          --fwidth;
        }
      }
    }

    
    // convert exponent

    char eneg = exp < 0;
    if (eneg) exp = - exp;

    while (exp > 0)
    {
      *--ew = '0' + (exp % 10);
      exp /= 10;
      ++ewidth;
    }

    while (ewidth < 2)  // ensure at least 2 zeroes
    {
      *--ew = '0';
      ++ewidth;
    }

    *--ew = eneg ? '-' : '+';
    *--ew = 'e';

    ewidth += 2;

    // convert the one-digit integer part
    *--iw = '0' + int(ipart);
    ++iwidth;
    
  }

  // arrange everything in returned string

  int showdot = cvt != 'g' || fwidth > 0;

  int fmtwidth = is_neg + iwidth + showdot + fwidth + ewidth;
  
  int pad = width - fmtwidth;
  if (pad < 0) pad = 0;
  
  char* fmtbase = _libgxx_fmtq.alloc(fmtwidth + pad + 1);
  char* fmt = fmtbase;
  
  for (int i = 0; i < pad; ++i) *fmt++ = ' ';
  
  if (is_neg) *fmt++ = '-';
  
  for (i = 0; i < iwidth; ++i) *fmt++ = *iw++;
  
  if (showdot)
  {
    *fmt++ = '.';
    fw = fwork;
    for (i = 0; i < fwidth; ++i) *fmt++ = *fw++;
  }
  
  for (i = 0; i < ewidth; ++i) *fmt++ = *ew++;
  
  *fmt = 0;
  
  return fmtbase;
}


static char chr_buf[2];   // fixed slot to hold chr(ch)

char* chr(char ch)
{
  chr_buf[0] = ch;
  chr_buf[1] = 0;
  return chr_buf;
}

char* str(const char* s, int width = 0)
{
  int len = strlen(s);
  int wrksiz = len + width;
  char* fmtbase = _libgxx_fmtq.alloc(wrksiz);
  char* fmt = fmtbase;
  for (int blanks = width - len; blanks > 0; --blanks)
   *fmt++ = ' ';
  while (*s != 0) 
    *fmt++ = *s++;
  *fmt = 0;
  return fmtbase;
}

/*
 some useful hash functions
*/

unsigned int hashpjw(const char* x) // From Dragon book, p436
{
  unsigned int h = 0;
  unsigned int g;

  while (*x != 0)
  {
    h = (h << 4) + *x++;
    if ((g = h & 0xf0000000) != 0)
      h = (h ^ (g >> 24)) ^ g;
  }
  return h;
}

unsigned int multiplicativehash(int x)
{
  // uses a const close to golden ratio * pow(2,32)
  return ((unsigned)x) * 2654435767;
}


unsigned int foldhash(double x)
{
  union { unsigned int i[2]; double d; } u;
  u.d = x;
  unsigned int u0 = u.i[0];
  unsigned int u1 = u.i[1]; 
  return u0 ^ u1;
}

void default_one_arg_error_handler(const char* msg)
{
  fputs("Error: ", stderr);
  fputs(msg, stderr);
  fputs("\n", stderr);
  abort();
}


void default_two_arg_error_handler(const char* kind, const char* msg)
{
  fputs(kind, stderr);
  fputs(" Error: ", stderr);
  fputs(msg, stderr);
  fputs("\n", stderr);
  abort();
}

two_arg_error_handler_t lib_error_handler = default_two_arg_error_handler;

two_arg_error_handler_t set_lib_error_handler(two_arg_error_handler_t f)
{
  two_arg_error_handler_t old = lib_error_handler;
  lib_error_handler = f;
  return old;
}


// from Doug Schmidt...

/* no such thing as "negative time"! */
#define  TIMER_ERROR_VALUE -1.0   

// surely OK for these machines...

#if defined(BSD) || defined(USG) || defined(vax) || defined(sun)

#ifdef USG
extern "C" {
#include <sys/types.h>
#include <sys/param.h>
#include <sys/times.h>
}
#else
#include <osfcn.h>
#endif

#ifdef USG
static struct tms Old_Time;
static struct tms New_Time;
#else
static struct rusage Old_Time;
static struct rusage New_Time;
#endif
static int    Timer_Set = 0;

double start_timer()
{
   Timer_Set = 1;
#ifdef USG
   times(&Old_Time);
   return((double) Old_Time.tms_utime / HZ);
#else
   getrusage(RUSAGE_SELF,&Old_Time);        /* set starting process time */
   return(Old_Time.ru_utime.tv_sec + (Old_Time.ru_utime.tv_usec / 1000000.0));
#endif
}

/* returns process time since Last_Time (if parameter is not DEFAULT_TIME, */
/* i.e., (double) 0.0 ),otherwise, if parameter == DEFAULT_TIME then       */
/* the time since the Old_Time was set is returned. */
/* Returns TIMER_ERROR_VALUE   */
/* if Start_Timer() is not called first */

double return_elapsed_time(double Last_Time)
{
   if (!Timer_Set) {
      return(TIMER_ERROR_VALUE);
   }   
   else {
    /* get process time */
#ifdef USG
      times(&New_Time);
#else
      getrusage(RUSAGE_SELF,&New_Time);
#endif
      if (Last_Time == 0.0) {
#ifdef USG
	 return((double) (New_Time.tms_utime - Old_Time.tms_utime) / HZ);
#else
         return((New_Time.ru_utime.tv_sec - Old_Time.ru_utime.tv_sec) + 
               ((New_Time.ru_utime.tv_usec - Old_Time.ru_utime.tv_usec) 
                / 1000000.0));
#endif
      }
      else {
#ifdef USG
	 return((double) New_Time.tms_utime / HZ - Last_Time);
#else
         return((New_Time.ru_utime.tv_sec + 
                (New_Time.ru_utime.tv_usec / 1000000.0)) - Last_Time);
#endif
      }
   }
}

#else /* dummy them out */
double start_timer()
{
  return TIMER_ERROR_VALUE;
}

double return_elapsed_time(double)
{
  return TIMER_ERROR_VALUE;
}

#endif


#ifndef NO_GNULIB3
// from tiemann

/* Needed, in case there are no other objects which
   need static initialization and cleanup.  */
struct __xyzzy__
{
  __xyzzy__ () {}
  ~__xyzzy__ () {}
} __1xyzzy__;

#endif
