// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

#ifndef _BitSet_h
#pragma once

#define _BitSet_h 1

#include <stream.h>
#include <values.h>

#define BITSETBITS  BITS(short)

struct BitSetRep
{
  unsigned short  len;          // number of shorts in s
  unsigned short  sz;           // allocated slots
  unsigned short  virt;         // virtual 0 or 1
  unsigned short  s[1];         // bits start here

  friend BitSetRep*   BitSetalloc(BitSetRep*, unsigned short*, int, int, int);
  friend BitSetRep*   BitSetcopy(BitSetRep*, BitSetRep*);
  friend BitSetRep*   BitSetresize(BitSetRep*, int);
  friend BitSetRep*   BitSetop(BitSetRep*, BitSetRep*, BitSetRep*, char);
  friend BitSetRep*   BitSetcmpl(BitSetRep*, BitSetRep*);

};

extern BitSetRep    _nilBitSetRep;
class BitSet;
class BitSetTmp;

class BitSetBit
{
protected:
  BitSet*            src;
  unsigned long      pos;

 public:
                     BitSetBit(BitSet* v, int p);
                     BitSetBit(BitSetBit& b);
                    ~BitSetBit();
                     operator int();
  int                operator = (int b);
  int                operator == (int b);
  int                operator != (int b);
};

class BitSet
{
  friend class       BitSetTmp;

protected:
  BitSetRep*          rep;


public:

// constructors
                     BitSet();
                     BitSet(BitSet&);

                     ~BitSet();

  void               operator =  (BitSet& y);
  void               operator =  (BitSetTmp& y);

// equality & subset tests

  friend int         operator == (BitSet& x, BitSet& y);
  friend int         operator != (BitSet& x, BitSet& y);
  friend int         operator <  (BitSet& x, BitSet& y);
  friend int         operator <= (BitSet& x, BitSet& y);
  friend int         operator >  (BitSet& x, BitSet& y);
  friend int         operator >= (BitSet& x, BitSet& y);

// set operators

  BitSetTmp          operator |  (BitSet& y);
  BitSetTmp          operator |  (BitSetTmp& y);
  void               operator |= (BitSet& y);
  BitSetTmp          operator &  (BitSet& y);
  BitSetTmp          operator &  (BitSetTmp& y);
  void               operator &= (BitSet& y);
  BitSetTmp          operator -  (BitSet& y);
  BitSetTmp          operator -  (BitSetTmp& y);
  void               operator -= (BitSet& y);
  BitSetTmp          operator ^  (BitSet& y);
  BitSetTmp          operator ^  (BitSetTmp& y);
  void               operator ^= (BitSet& y);

  BitSetTmp          operator ~  ();
  void               complement();

// individual bit manipulation

  void               set(int pos);
  void               set(int from, int to);
  void               set(); // set all

  void               clear(int pos);
  void               clear(int from, int to);
  void               clear(); // clear all

  void               invert(int pos);
  void               invert(int from, int to);

  int                test(int pos);
  int                test(int from, int to);

  BitSetBit          operator [] (int i);
  
// iterators

  int                first(int b = 1);
  int                last(int b = 1);

  int                next(int pos, int b = 1);
  int                previous(int pos, int b = 1);

// status

  int                empty();
  int                virtual_bit();
  int                count(int b = 1);
  
// convertors & IO

  friend BitSetTmp   atoBitSet(const char* s, 
                               char f='0', char t='1', char star='*');
  friend const char* BitSettoa(BitSet& x, 
                               char f='0', char t='1', char star='*');

  friend BitSetTmp   shorttoBitSet(unsigned short w);
  friend BitSetTmp   longtoBitSet(unsigned long w);

  friend ostream&    operator << (ostream& s, BitSet& x);

// misc

  void               error(char* msg);
  int                OK();
};

class BitSetTmp: public BitSet
{
  friend class       BitSet;
public:
                     BitSetTmp(BitSetRep*);
                     BitSetTmp(BitSet& x);
                     BitSetTmp(BitSetTmp& x);
                     ~BitSetTmp();

  BitSetTmp          operator |  (BitSet& y);
  BitSetTmp          operator &  (BitSet& y);
  BitSetTmp          operator -  (BitSet& y);
  BitSetTmp          operator ^  (BitSet& y);

  BitSetTmp          operator ~  ();
};

//#ifdef __OPTIMIZE__

inline int BitSet_index(int l)
{
  return (unsigned)(l) / BITSETBITS;
}

inline int BitSet_pos(int l)
{
  return l & (BITSETBITS - 1);
}

inline BitSet::BitSet()
{ 
  rep = &_nilBitSetRep;
}

inline BitSet::BitSet(BitSet& x)
{ 
  rep = BitSetcopy(0, x.rep);
}

inline BitSetTmp::BitSetTmp(BitSetRep* x) 
{
  rep = x;
}

inline BitSetTmp::BitSetTmp(BitSet& x)
{
  rep = x.rep; x.rep = &_nilBitSetRep;
}

inline BitSetTmp::BitSetTmp(BitSetTmp& x)
{
  rep = x.rep; x.rep = &_nilBitSetRep;
}

inline BitSet::~BitSet()
{ 
  if (rep != &_nilBitSetRep) delete rep;
}

inline BitSetTmp::~BitSetTmp() {}

inline void BitSet::operator =  (BitSet& y)
{ 
  rep = BitSetcopy(rep, y.rep);
}

inline void BitSet::operator =  (BitSetTmp& y)
{ 
  if (rep != &_nilBitSetRep) delete rep;
  rep = y.rep; y.rep = &_nilBitSetRep;
}

inline int operator != (BitSet& x, BitSet& y)
{
  return !(x == y);
}

inline int operator>(BitSet& x, BitSet& y)
{
  return y < x;
}

inline int operator>=(BitSet& x, BitSet& y)
{
  return y <= x;
}

inline BitSetTmp BitSet::operator & (BitSet& y)
{
  return  BitSetop(rep, y.rep, 0, '&');
}

inline BitSetTmp BitSet::operator & (BitSetTmp& y)
{
  y.rep = BitSetop(rep, y.rep, y.rep, '&'); return y;
}

inline BitSetTmp BitSetTmp::operator & (BitSet& y)
{
  rep = BitSetop(rep, y.rep, rep, '&'); return *this;
}

inline void BitSet::operator &= (BitSet& y)
{
  rep = BitSetop(rep, y.rep, rep, '&'); 
}

inline BitSetTmp BitSet::operator | (BitSet& y)
{
  return  BitSetop(rep, y.rep, 0, '|');
}

inline BitSetTmp BitSet::operator | (BitSetTmp& y)
{
  y.rep = BitSetop(rep, y.rep, y.rep, '|'); return y;
}

inline BitSetTmp BitSetTmp::operator | (BitSet& y)
{
  rep = BitSetop(rep, y.rep, rep, '|'); return *this;
}

inline void BitSet::operator |= (BitSet& y)
{
  rep = BitSetop(rep, y.rep, rep, '|'); 
}

inline BitSetTmp BitSet::operator ^ (BitSet& y)
{
  return  BitSetop(rep, y.rep, 0, '^');
}

inline BitSetTmp BitSet::operator ^ (BitSetTmp& y)
{
  y.rep = BitSetop(rep, y.rep, y.rep, '^'); return y;
}

inline BitSetTmp BitSetTmp::operator ^ (BitSet& y)
{
  rep = BitSetop(rep, y.rep, rep, '^'); return *this;
}

inline void BitSet::operator ^= (BitSet& y)
{
  rep = BitSetop(rep, y.rep, rep, '^'); 
}

inline BitSetTmp BitSet::operator - (BitSet& y)
{
  return  BitSetop(rep, y.rep, 0, '-');
}

inline BitSetTmp BitSet::operator - (BitSetTmp& y)
{
  y.rep = BitSetop(rep, y.rep, y.rep, '-'); return y;
}

inline BitSetTmp BitSetTmp::operator - (BitSet& y)
{
  rep = BitSetop(rep, y.rep, rep, '-'); return *this;
}

inline void BitSet::operator -= (BitSet& y)
{
  rep = BitSetop(rep, y.rep, rep, '-'); 
}

inline BitSetTmp BitSet::operator ~ ()
{
  return BitSetcmpl(rep, 0);
}

inline BitSetTmp BitSetTmp::operator ~ ()
{
  rep = BitSetcmpl(rep, rep); return *this;
}

inline void BitSet::complement()
{
  rep = BitSetcmpl(rep, rep);
}

inline int BitSet::virtual_bit()
{
  return rep->virt;
}

inline int BitSet::first(int b = 1)
{
  return next(-1, b);
}

inline int BitSet::test(int p)
{
  if (p < 0) error("Illegal bit index");
  int index = BitSet_index(p);
  return (index >= rep->len)? rep->virt : 
         ((rep->s[index] & (1 << BitSet_pos(p))) != 0);
}


inline void BitSet::clear()
{
  if (rep->len > 0) bzero(rep->s, rep->sz * sizeof(short));
  rep->len = rep->virt = 0;
}

inline void BitSet::set()
{
  rep = BitSetalloc(rep, 0, 0, 1, 0);
}

inline BitSetBit::BitSetBit(BitSetBit& b)
{
  src = b.src;  pos = b.pos;
}

inline BitSetBit::BitSetBit(BitSet* v, int p)
{
  src = v;  pos = p;
}

inline BitSetBit::~BitSetBit() {}

inline BitSetBit::operator int()
{
  return src->test(pos);
}

inline int BitSetBit::operator = (int b)
{
  if (b) src->set(pos); else src->clear(pos); return b;
}

inline int BitSetBit::operator == (int b)
{
  return src->test(pos) == b;
}

inline int BitSetBit::operator != (int b)
{
  return src->test(pos) != b;
}

inline BitSetBit BitSet::operator [] (int i)
{
  if (i < 0) error("illegal bit index");
  return BitSetBit(this, i);
}


//#endif

#endif
