/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

/* 
  BitString class implementation
 */

#include <BitString.h>
#include <std.h>
#include "libconfig.h"

// error handling


void default_BitString_error_handler(char* msg)
{
  cerr << "Fatal BitString error. " << msg << "\n";
  exit(1);
}

one_arg_error_handler_t BitString_error_handler = default_BitString_error_handler;

one_arg_error_handler_t set_BitString_error_handler(one_arg_error_handler_t f)
{
  one_arg_error_handler_t old = BitString_error_handler;
  BitString_error_handler = f;
  return old;
}

void BitString::error(char* msg)
{
  (*BitString_error_handler)(msg);
}

//  globals

_BitStringrep    _nil_BitStringrep = {  0, 1, -1, {0} };

static BitString _nil_BitString;


#define MIN_BITSTRING_SIZE  2
#define MAX_BITSTRING_SIZE  (1 << (SHORTBITS - 1) - 1)

#define ONES    (~(0L))
#define MAXBIT  (1 << (B_SHIFT - 1))

// allocate a new rep

static _BitStringrep*  new_BitStringrep(int l)
{
  int siz = l / B_SHIFT + 1;
  if (siz < MIN_BITSTRING_SIZE)
    siz = MIN_BITSTRING_SIZE;
  else if (siz >= MAX_BITSTRING_SIZE)
    (*BitString_error_handler)("Requested length out of range");

  unsigned allocsiz = sizeof(_BitStringrep) + (siz - 1) * sizeof(long);

  _BitStringrep* z = (_BitStringrep *) (malloc(allocsiz));
  if (z == 0)
    (*BitString_error_handler)("Out of memory.");

  z->len = l;
  z->sz  = siz;
  z->ref = 1;
  return z;
}

// expand a rep via realloc

static _BitStringrep*  expand_BitStringrep(_BitStringrep* p, int l)
{
  int siz = l / B_SHIFT + 1;    
  if (siz < MIN_BITSTRING_SIZE)   
    siz = MIN_BITSTRING_SIZE;
  else if (siz >= MAX_BITSTRING_SIZE)
    (*BitString_error_handler)("Requested length out of range");

  unsigned allocsiz = sizeof(_BitStringrep) + (siz - 1) * sizeof(long);

#ifdef SHOULD_FREE_TO_REALLOC
  free((void*)p);
#endif

  _BitStringrep* z = (_BitStringrep *) (realloc((void*)p, allocsiz));
  if (z == 0)
    (*BitString_error_handler)("Out of memory.");
  z->len = l;
  z->sz  = siz;
  z->ref = 1;
  return z;
}

// mask out low bits

static inline unsigned long lmask(int p)
{
  if (p <= 0)
    return ~(0L);
  else
    return ((~(0L)) << p);
}

// mask out high bits

static inline unsigned long rmask(int p)
{
  int s = p + 1;
  if (s >= B_SHIFT)
    return ~(0L);
  else
    return (1 << s) - 1;
}


inline static int nwords(int l)
{
  return l / B_SHIFT + 1;
}

inline static int nlast(int l)
{
  return l % B_SHIFT;
}

// mask out top bits

inline static void check_last(_BitStringrep* r)
{
  int l = r->len / B_SHIFT;
  int last = r->len % B_SHIFT;
  if (last < B_SHIFT - 1)
    r->s[l] &= (1 << last) - 1;
}

// merge bits from next word

static inline unsigned long borrow_hi(unsigned long a[], int ind, 
                                      int maxind, int p)
{
  if (ind > maxind)
    return 0;
  else if (p == 0)
    return a[ind];
  else if (ind < maxind)
    return (((a[ind] >> p) & ((1 << (B_SHIFT - p)) - 1)) 
            | (a[ind+1] << (B_SHIFT - p)));
  else
    return (((a[ind] >> p) & ((1 << (B_SHIFT - p)) - 1)));
}

// merge bits from prev word

static inline unsigned long borrow_lo(unsigned long a[], int ind, 
                                      int minind, int p)
{
  if (ind < minind)
    return 0;
  else if (p == B_SHIFT - 1)
    return a[ind];
  else if (ind > minind)
    return (((a[ind-1] >> p+1) & ((1 << (B_SHIFT - p)) - 1)) | 
            (a[ind] << (B_SHIFT - 1 - p)));
  else
    return a[ind] << (B_SHIFT - 1 - p);
}

static inline void bit_copy(unsigned long* ss, unsigned long* ds, int nbits)
{
  if (ss != ds)
  {
    int n = nbits / B_SHIFT;
    if (n > 0) bcopy((void*)ss, (void*)ds, n * sizeof(long));
    unsigned long m = ~(0L) << (nbits % B_SHIFT);
    ds[n] = (ss[n] & ~m) | (ds[n] & m);
  }
}

// clear bits from a word boundary

static inline void bit_clear(unsigned long* ds, int nbits)
{
  int n = nbits / B_SHIFT;
  int l = nbits % B_SHIFT;
  if (n > 0) bzero((void*)ds, n * sizeof(long));
  ds[n] &= ~(0L) << l;
}


// copy ss from starts to ends-1 into ds starting at startd
// this will work even if ss & ds are the same 
// This works right now, but is far from optimal...

static void bit_transfer(unsigned long* ss, int starts, int lengths,
                         unsigned long* ds, int startd)
{
  int ends = lengths - 1;
  if (starts > ends)
    return;

  int sind = starts / B_SHIFT;
  int spos = starts % B_SHIFT;
  int endsind = ends / B_SHIFT;
  int endspos = ends % B_SHIFT;
  int dind = startd / B_SHIFT;
  int dpos = startd % B_SHIFT;
  int endd = startd + (ends - starts);
  int enddind = endd / B_SHIFT;
  int enddpos = endd % B_SHIFT;
  
  unsigned long lodmask = (1 << dpos) - 1;
  unsigned long hidmask = ONES << (enddpos + 1);

  unsigned long a;
  if (dind == enddind)
  {
    a = borrow_hi(ss, sind, endsind, spos);
    a  = (a << dpos) & ~(lodmask | hidmask);
    ds[dind] = (ds[dind] & (lodmask | hidmask)) | a;
  }
  else if (ds != ss || startd < starts)
  {
    unsigned long savea = borrow_lo(ss, endsind, sind, endspos);
    savea = (savea >> (B_SHIFT - 1 - enddpos)) & ~hidmask;

    a = borrow_hi(ss, sind, endsind, spos);
    a = (a << dpos) & ~lodmask;
    unsigned long saved = (ds[dind] & lodmask) | a;

    if (dpos == 0)
    {
      ds[dind] = saved;
      while (++dind < enddind)
        ds[dind] = ss[++sind];
    }
    else
    {
      int pos = B_SHIFT - dpos;
      while (++dind < enddind)
      {
        unsigned long tmp = borrow_hi(ss, sind++, endsind, pos);
        ds[dind-1] = saved;
        saved = tmp;
      }
      ds[dind-1] = saved;
    }
    ds[enddind] = (ds[enddind] & hidmask) | savea;
  }
  else
  {
    unsigned long savea = borrow_hi(ss, sind, endsind, spos);
    savea = (savea << dpos) & ~lodmask;

    a = borrow_lo(ss, endsind, sind, endspos);
    a = (a >> (B_SHIFT - 1 - enddpos)) & ~hidmask;
    unsigned long saved = (ds[enddind] & hidmask) | a;

    if (enddpos == 0)
    {
      ds[enddind] = saved;
      while (--enddind > dind)
        ds[enddind] = ss[--endsind];
    }
    else
    {
      int pos = B_SHIFT - 1 - enddpos;
      while (--enddind > dind)
      {
        unsigned long tmp = borrow_lo(ss, endsind--, sind, pos);
        ds[enddind+1] = saved;
        saved = tmp;
      }
      ds[enddind+1] = saved;
    }
    ds[dind] = (ds[dind] & lodmask) | savea;
  }
}
  

// set up or expand a rep 

void BitString::setlength(int newlen, int prefill = 0)
{
  check_last(rep);
  int oldlen = rep->len;

  if (rep->ref != 1)
  {
    if (rep->ref > 0) rep->ref--;
    _BitStringrep* newrep = new_BitStringrep(newlen);
    bit_copy(rep->s, newrep->s, oldlen);
    rep = newrep;
  }
  else if (newlen > rep->sz * B_SHIFT)
    rep = expand_BitStringrep(rep, newlen);
  else
    rep->len = newlen;

  if (prefill && newlen > oldlen)
  {
    int ol = nwords(oldlen);
    int nl = nwords(newlen);
    bzero((void*)&(rep->s[ol]), (nl - ol) * sizeof(long));
    check_last(rep);
  }
}


// copy one rep to another; expand & dereference as necessary

void BitString::copy(unsigned long* news, int startpos, int endp)
{
  int newlen = endp - startpos;
  if (newlen <= 0)
  {
    if (rep->ref > 0 && --rep->ref == 0)
      delete rep;
    rep = &_nil_BitStringrep;
    return;
  }

  if (rep->ref != 1)
  {
    if (rep->ref > 0) rep->ref--;
    rep = new_BitStringrep(newlen);
  }
  else if (newlen > rep->sz * B_SHIFT)
    rep = expand_BitStringrep(rep, newlen);

  bit_transfer(news, startpos, endp, rep->s, 0);
  rep->len = newlen;
  check_last(rep);
}

int operator == (BitString& x, BitString& y)
{
  return x.rep->len == y.rep->len && 
    bcmp((void*)x.rep->s, (void*)y.rep->s, 
         nwords(x.rep->len) * sizeof(long)) == 0;
}

int operator <= (BitString& x, BitString& y)
{
  int xl = x.rep->len;
  int yl = y.rep->len;
  if (xl > yl)
    return 0;

  unsigned long* xs = x.rep->s;
  unsigned long* ys = y.rep->s;
  unsigned long* topx = &(xs[nwords(xl)]);

  while (xs < topx)
  {
    unsigned long a = *xs++;
    unsigned long b = *ys++;
    if ((a | b) != b)
      return 0;
  }
  return 1;
}

int operator < (BitString& x, BitString& y)
{
  int xl = x.rep->len;
  int yl = y.rep->len;
  if (xl > yl)
    return 0;

  unsigned long* xs = x.rep->s;
  unsigned long* ys = y.rep->s;
  unsigned long* topx = &(xs[nwords(xl)]);
  unsigned long* topy = &(xs[nwords(yl)]);
  int one_diff = 0;
  while (xs < topx)
  {
    unsigned long a = *xs++;
    unsigned long b = *ys++;
    unsigned long c = a | b;
    if (c != b)
      return 0;
    else if (c != a)
      one_diff = 1;
  }
  if (one_diff)
    return 1;
  else
  {
    while (ys < topy)
      if (*ys != 0)
        return 1;
    return 0;
  }
}

int lcompare(BitString& x, BitString& y)
{
  int xl = x.rep->len;
  int yl = y.rep->len;

  unsigned long* xs = x.rep->s;
  unsigned long* topx = &(xs[nwords(xl)]);
  unsigned long* ys = y.rep->s;
  unsigned long* topy = &(ys[nwords(yl)]);

  while (xs < topx && ys < topy)
  {
    unsigned long a = *xs++;
    unsigned long b = *ys++;
    if (a != b)
    {
      unsigned long mask = 1;
      for (;;)
      {
        unsigned long abit = (a & mask) != 0;
        unsigned long bbit = (b & mask) != 0;
        int diff = abit - bbit;
        if (diff != 0)
          return diff;
        else
          mask <<= 1;
      }
    }
  }
  return xl - yl;
}

int BitString::count(int b = 1)
{
  check_last(rep);
  int xwds = nwords(rep->len);
  int xlast = nlast(rep->len);
  int l = 0;
  unsigned long* s = rep->s;
  unsigned long* tops = &(s[xwds - 1]);
  unsigned long a;
  int i;
  if (b == 1)
  {
    while (s < tops)
    {
      a = *s++;
      for (i = 0; i < B_SHIFT && a != 0; ++i)
      {
        if (a & 1)
          ++l;
        a >>= 1;
      }
    }
    a = *s;
    for (i = 0; i < xlast && a != 0; ++i)
    {
      if (a & 1)
        ++l;
      a >>= 1;
    }
  }
  else
  {
    unsigned long maxbit = 1 << (B_SHIFT - 1);
    while (s < tops)
    {
      a = *s++;
      for (i = 0; i < B_SHIFT; ++i)
      {
        if ((a & maxbit) == 0)
          ++l;
        a <<= 1;
      }
    }
    maxbit = 1 << (xlast - 1);
    a = *s;
    for (i = 0; i < xlast; ++i)
    {
      if ((a & maxbit) == 0)
        ++l;
      a <<= 1;
    }
  }
  return l;
}


BitString BitString:: operator ~()
{
  BitString r;
  r.setlength(rep->len);
  unsigned long* xs = rep->s;
  unsigned long* rs = r.rep->s;
  unsigned long* topr = &(rs[nwords(rep->len)]);
  while (rs < topr) *rs++ = ~(*xs++);
  return r;
}


BitString& BitString::complement()
{
  make_unique();
  unsigned long* rs = rep->s;
  unsigned long* topr = &(rs[nwords(rep->len)]);
  while (rs < topr)
  {
    unsigned long cmp = ~(*rs);
    *rs++ = cmp;
  }
  return *this;
}

void and(BitString& x, BitString& y, BitString& r)
{
  int xl = x.rep->len;
  int yl = y.rep->len;
  int rl = xl <? yl;
  int xrsame = x.rep == r.rep;
  int yrsame = y.rep == r.rep;

  r.setlength(rl);

  unsigned long* rs = r.rep->s;
  unsigned long* topr = &(rs[nwords(rl)]);
  unsigned long* xs = (xrsame)? rs : x.rep->s;
  unsigned long* ys = (yrsame)? rs : y.rep->s;

  while (rs < topr) *rs++ = *xs++ & *ys++;
  check_last(r.rep);
}

void or(BitString& x, BitString& y, BitString& r)
{
  int xl = x.rep->len;
  int yl = y.rep->len;
  int rl = xl >? yl;
  int xrsame = x.rep == r.rep;
  int yrsame = y.rep == r.rep;

  r.setlength(rl);

  unsigned long* rs = r.rep->s;
  unsigned long* xs = (xrsame)? rs : x.rep->s;
  unsigned long* topx = &(xs[nwords(xl)]);
  unsigned long* ys = (yrsame)? rs : y.rep->s;
  unsigned long* topy = &(ys[nwords(yl)]);

  if (xl <= yl)
  {
    while (xs < topx) *rs++ = *xs++ | *ys++;
    if (rs != ys) while (ys < topy) *rs++ = *ys++;
  }
  else
  {
    while (ys < topy) *rs++ = *xs++ | *ys++;
    if (rs != xs) while (xs < topx) *rs++ = *xs++;
  }
  check_last(r.rep);
}


void xor(BitString& x, BitString& y, BitString& r)
{
  int xl = x.rep->len;
  int yl = y.rep->len;
  int rl = xl >? yl;
  int xrsame = x.rep == r.rep;
  int yrsame = y.rep == r.rep;

  r.setlength(rl);

  unsigned long* rs = r.rep->s;
  unsigned long* xs = (xrsame)? rs : x.rep->s;
  unsigned long* topx = &(xs[nwords(xl)]);
  unsigned long* ys = (yrsame)? rs : y.rep->s;
  unsigned long* topy = &(ys[nwords(yl)]);

  if (xl <= yl)
  {
    while (xs < topx) *rs++ = *xs++ ^ *ys++;
    if (rs != ys) while (ys < topy) *rs++ = *ys++;
  }
  else
  {
    while (ys < topy) *rs++ = *xs++ ^ *ys++;
    if (rs != xs) while (xs < topx) *rs++ = *xs++;
  }
  check_last(r.rep);
}


void difference(BitString& x, BitString& y, BitString& r)
{
  int xl = x.rep->len;
  int yl = y.rep->len;
  int xrsame = x.rep == y.rep;
  int yrsame = y.rep == r.rep;

  r.setlength(xl);

  unsigned long* rs = r.rep->s;
  unsigned long* xs = (xrsame)? rs : x.rep->s;
  unsigned long* topx = &(xs[nwords(xl)]);
  unsigned long* ys = (yrsame)? rs : y.rep->s;
  unsigned long* topy = &(ys[nwords(yl)]);

  if (xl <= yl)
  {
    while (xs < topx) *rs++ = *xs++ & ~(*ys++);
  }
  else
  {
    while (ys < topy) *rs++ = *xs++ & ~(*ys++);
    if (rs != xs) while (xs < topx) *rs++ = *xs++;
  }
  check_last(r.rep);
}


void concat(BitString& x, BitString& y, BitString& r)
{
  int xl = x.rep->len;
  int xwrds = nwords(xl);
  int yl = y.rep->len;
  int rl = xl + yl;
  int xrsame = x.rep == r.rep;
  int yrsame = y.rep == r.rep;

  if (yrsame)
  {
    BitString tmp = y;
    r.setlength(rl);
    bit_copy(x.rep->s, r.rep->s, xl);
    bit_transfer(tmp.rep->s, 0, yl, r.rep->s, xl);
  }
  else
  {
    r.setlength(rl);
    bit_copy(x.rep->s, r.rep->s, xl);
    bit_transfer(y.rep->s, 0, yl, r.rep->s, xl);
  }
  check_last(r.rep);
}

void rshift(BitString& x, int s, BitString& r)
{
  if (s == 0)
  {
    r = x;
    return;
  }

  int xl = x.rep->len;
  int rl = xl + s;
  if (rl <= 0)
  {
    r = _nil_BitString;
    return;
  }

  int xrsame = x.rep == r.rep;
  r.setlength(rl);
  unsigned long* xs = (xrsame)? r.rep->s : x.rep->s;

  if (s < 0)
    bit_transfer(xs, -s, xl, r.rep->s, 0);
  else
  {
    bit_transfer(xs, 0, xl, r.rep->s, s);
    bit_clear(r.rep->s, s);
  }

  check_last(r.rep);
}


void BitString::set(int p)
{
  if (p < 0)
  {
    error("Illegal bit index");
    return;
  }
  
  if (p >= rep->len)
    setlength(p + 1, 1);
  else
    make_unique();

  rep->s[p / B_SHIFT] |= (1 << (p % B_SHIFT));
}

void BitString::clear(int p)
{
  if (p < 0)
  {
    error("Illegal bit index");
    return;
  }
  
  if (p >= rep->len)
    setlength(p + 1, 1);
  else
    make_unique();
  
  rep->s[p / B_SHIFT] &= ~(1 << (p % B_SHIFT));
}

void BitString::invert(int p)
{
  if (p < 0)
  {
    error("Illegal bit index");
    return;
  }

  if (p >= rep->len)
    setlength(p + 1, 1);
  else
    make_unique();
  
  rep->s[p / B_SHIFT] ^= (1 << (p % B_SHIFT));
}

int BitString::test(int p)
{
  if (p < 0 || p >= rep->len)
    return 0;
  else
    return (rep->s[p / B_SHIFT] & (1 << (p % B_SHIFT))) != 0;
}


void BitString::set(int from, int to)
{
  if (from < 0 || from > to)
  {
    error("Illegal bit index");
    return;
  }

  int ind1 = from / B_SHIFT;
  int pos1 = from % B_SHIFT;
  int ind2 = to / B_SHIFT;
  int pos2 = to % B_SHIFT;

  if (to >= rep->len)
    setlength(to+1, 1);
  else
    make_unique();

  unsigned long* s = &(rep->s[ind1]);
  unsigned long m1 = lmask(pos1);
  unsigned long m2 = rmask(pos2);
  if (ind2 == ind1)
    *s |= m1 & m2;
  else
  {
    *s++ |= m1;
    unsigned long* top = &(rep->s[ind2]);
    *top |= m2;
    while (s < top)
      *s++ = ONES;
  }
}

void BitString::clear(int from, int to)
{
  if (from < 0 || from > to)
  {
    error("Illegal bit index");
    return;
  }

  int ind1 = from / B_SHIFT;
  int pos1 = from % B_SHIFT;
  int ind2 = to / B_SHIFT;
  int pos2 = to % B_SHIFT;

  if (to >= rep->len)
    setlength(to+1, 1);
  else
    make_unique();

  unsigned long* s = &(rep->s[ind1]);
  unsigned long m1 = lmask(pos1);
  unsigned long m2 = rmask(pos2);
  if (ind2 == ind1)
    *s &= ~(m1 & m2);
  else
  {
    *s++ &= ~m1;
    unsigned long* top = &(rep->s[ind2]);
    *top &= ~m2;
    while (s < top)
      *s++ = 0;
  }
}

void BitString::invert(int from, int to)
{
  if (from < 0 || from > to)
  {
    error("Illegal bit index");
    return;
  }

  int ind1 = from / B_SHIFT;
  int pos1 = from % B_SHIFT;
  int ind2 = to / B_SHIFT;
  int pos2 = to % B_SHIFT;

  if (to >= rep->len)
    setlength(to+1, 1);
  else
    make_unique();

  unsigned long* s = &(rep->s[ind1]);
  unsigned long m1 = lmask(pos1);
  unsigned long m2 = rmask(pos2);
  if (ind2 == ind1)
    *s ^= m1 & m2;
  else
  {
    *s++ ^= m1;
    unsigned long* top = &(rep->s[ind2]);
    *top ^= m2;
    while (s < top)
    {
      unsigned long cmp = ~(*s);
      *s++ = cmp;
    }
  }
}


int BitString::test(int from, int to)
{
  if (from < 0 || from > to)
  {
    return 0;
  }

  int ind1 = from / B_SHIFT;
  int pos1 = from % B_SHIFT;
  
  if (from >= rep->len)
    return 0;

  int ind2 = to / B_SHIFT;
  int pos2   = to % B_SHIFT;

  if (to >= rep->len)
  {
    ind2 = (rep->len - 1) / B_SHIFT;
    pos2 = (rep->len - 1) % B_SHIFT;
  }

  unsigned long* s = &(rep->s[ind1]);
  unsigned long m1 = lmask(pos1);
  unsigned long m2 = rmask(pos2);

  if (ind2 == ind1)
    return (*s & m1 & m2) != 0;
  else
  {
    if (*s++ & m1)
      return 1;
    unsigned long* top = &(rep->s[ind2]);
    if (*top & m2)
      return 1;
    while (s < top)
      if (*s++ != 0) 
        return 1;
    return 0;
  }
}

int BitString::next(int p, int b = 1)
{
  if (++p >= rep->len)
    return -1;
  int ind = p / B_SHIFT;
  int pos = p % B_SHIFT;

  int l = nwords(rep->len);

  int j = ind;
  unsigned long* s = rep->s;
  unsigned long a = s[j] >> pos;
  int i = pos;

  if (b == 1)
  {
    for (; i < B_SHIFT && a != 0; ++i)
    {
      if (a & 1)
        return j * B_SHIFT + i;
      a >>= 1;
    }
    for (++j; j < l; ++j)
    {
      a = s[j];
      for (i = 0; i < B_SHIFT && a != 0; ++i)
      {
        if (a & 1)
          return j * B_SHIFT + i;
        a >>= 1;
      }
    }
    return -1;
  }
  else
  {
    int last = nlast(rep->len);
    if (j == l - 1)
    {
      for (; i < last; ++i)
      {
        if ((a & 1) == 0)
          return j * B_SHIFT + i;
        a >>= 1;
      }
      return -1;
    }

    for (; i < B_SHIFT; ++i)
    {
      if ((a & 1) == 0)
        return j * B_SHIFT + i;
      a >>= 1;
    }
    for (++j; j < l - 1; ++j)
    {
      a = s[j];
      if (a != ONES)
      {
        for (i = 0; i < B_SHIFT; ++i)
        {
          if ((a & 1) == 0)
            return j * B_SHIFT + i;
          a >>= 1;
        }
      }
    }
    a = s[j];
    for (i = 0; i < last; ++i)
    {
      if ((a & 1) == 0)
        return j * B_SHIFT + i;
      a >>= 1;
    }
    return -1;
  }
}

int BitString::previous(int p, int b = 1)
{
  if (--p < 0)
    return -1;

  int ind = p / B_SHIFT;
  int pos = p % B_SHIFT;

  unsigned long* s = rep->s;
  int l = nwords(rep->len);

  if (p >= rep->len)
  {
    ind = (rep->len - 1) / B_SHIFT;
    pos = (rep->len - 1) % B_SHIFT;
  }

  int j = ind;
  unsigned long a = s[j];

  int i = pos;
  unsigned long maxbit = 1 << pos;

  if (b == 1)
  {
    for (; i >= 0 && a != 0; --i)
    {
      if (a & maxbit)
        return j * B_SHIFT + i;
      a <<= 1;
    }
    maxbit = 1 << (B_SHIFT - 1);
    for (--j; j >= 0; --j)
    {
      a = s[j];
      for (i = B_SHIFT - 1; i >= 0 && a != 0; --i)
      {
        if (a & maxbit)
          return j * B_SHIFT + i;
        a <<= 1;
      }
    }
    return -1;
  }
  else
  {
    if (a != ONES)
    {
      for (; i >= 0; --i)
      {
        if ((a & maxbit) == 0)
          return j * B_SHIFT + i;
        a <<= 1;
      }
    }
    maxbit = 1 << (B_SHIFT - 1);
    for (--j; j >= 0; --j)
    {
      a = s[j];
      if (a != ONES)
      {
        for (i = B_SHIFT - 1; i >= 0; --i)
        {
          if ((a & maxbit) == 0)
            return j * B_SHIFT + i;
          a <<= 1;
        }
      }
    }
    return -1;
  }
}


int BitString::index(int bit, int startpos = 0)
{
  if (startpos >= 0)
    return next(startpos - 1, bit);
  else
    return previous(rep->len + startpos + 1, bit);
}



int BitString::search(int startx, int lengthx, 
                      unsigned long* ys, int starty, int lengthy)
{
  unsigned long* xs = rep->s;
  int ylen = lengthy - starty;
  int righty = lengthy - 1;
  int leftx, rightx;
  int rev = startx < 0;
  if (rev)
  {
    leftx = 0;
    rightx = lengthx + startx;
    startx = rightx - ylen + 1;
  }
  else
  {
    leftx = startx;
    rightx = lengthx - 1;
  }

  if (starty < 0 || righty < 0 || startx < 0 || startx >= lengthx)
    return -1;
  
  int xind = startx / B_SHIFT;
  int xpos = startx % B_SHIFT;
  int yind = starty / B_SHIFT;
  int ypos = starty % B_SHIFT;

  int rightxind = rightx / B_SHIFT;
  int leftxind = leftx / B_SHIFT;
  unsigned long x = borrow_hi(xs, xind, rightxind, xpos);
  unsigned long nextx;
  if (rev)
    nextx = xs[xind] << (B_SHIFT - xpos);
  else
    nextx = (xind >= rightxind) ? 0 : (xs[xind+1] >> xpos);
  
  int rightyind = righty / B_SHIFT;
  int rightypos = righty % B_SHIFT;
  unsigned long y = borrow_hi(ys, yind, rightyind, ypos);
  unsigned long ymask;
  if (yind == rightyind)
    ymask = rmask(rightypos);
  else if (yind+1 == rightyind)
    ymask = rmask(ypos + rightypos + 1);
  else
    ymask = ONES;
  
  int p = startx;
  for (;;)
  {
    if ((x & ymask) == y)
    {
      int xi = xind;
      int yi = yind;
      for (;;)
      {
        if (++yi > rightyind || ++xi > rightxind)
          return p;
        unsigned long tx = borrow_hi(xs, xi, rightxind, xpos);
        unsigned long ty = borrow_hi(ys, yi, rightyind, ypos);
        unsigned long tm;
        if (yi == rightyind)
          tm = rmask(rightypos);
        else if (yi+1 == rightyind)
          tm = rmask(ypos + rightypos + 1);
        else
          tm = ONES;
        if ((tx & tm) != ty)
          break;
      }
    }
    if (rev)
    {
      if (--p < leftx)
        return -1;
      if (--xpos < 0)
      {
        xpos = B_SHIFT - 1;
        x = xs[--xind];
        nextx = (xind <= leftxind) ? 0 : xs[xind-1];
      }
      else
      {
        x <<= 1;
        if (nextx & MAXBIT)
          x |= 1;
        nextx <<= 1;
      }
    }
    else
    {
      if (++p > rightx)
        return -1;
      if (++xpos == B_SHIFT)
      {
        xpos = 0;
        x = xs[++xind];
        nextx = (xind >= rightxind) ? 0 : xs[xind+1];
      }
      else
      {
        x >>= 1;
        if (nextx & 1)
          x |= MAXBIT;
        else
          x &= ~MAXBIT;
        nextx >>= 1;
      }
    }
  }
}


int BitPattern::search(unsigned long* xs, int startx, int lengthx)
{
  unsigned long* ys = pattern.rep->s;
  unsigned long* ms = mask.rep->s;
  int righty = pattern.rep->len - 1;
  int rightm = mask.rep->len - 1;

  int leftx, rightx;
  int rev = startx < 0;
  if (rev)
  {
    leftx = 0;
    rightx = lengthx + startx;
    startx = rightx - righty;
  }
  else
  {
    leftx = startx;
    rightx = lengthx - 1;
  }

  if (righty < 0 || startx < 0 || startx >= lengthx)
    return -1;
  
  int xind = startx / B_SHIFT;
  int xpos = startx % B_SHIFT;

  int rightxind = rightx / B_SHIFT;
  int leftxind = leftx / B_SHIFT;
  int rightmind = rightm / B_SHIFT;
  int rightyind = righty / B_SHIFT;

  unsigned long x = borrow_hi(xs, xind, rightxind, xpos);
  unsigned long m = borrow_hi(ms, 0, rightmind, 0);
  unsigned long y = borrow_hi(ys, 0, rightyind, 0) & m;

  unsigned long nextx;
  if (rev)
    nextx = xs[xind] << (B_SHIFT - xpos);
  else
    nextx = (xind >= rightxind) ? 0 : (xs[xind+1] >> xpos);

  int p = startx;
  for (;;)
  {
    if ((x & m) == y)
    {
      int xi = xind;
      int yi = 0;
      for (;;)
      {
        if (++yi > rightyind || ++xi > rightxind)
          return p;
        unsigned long tm = borrow_hi(ms, yi, rightmind, 0);
        unsigned long tx = borrow_hi(xs, xi, rightxind, xpos) & tm;
        unsigned long ty = borrow_hi(ys, yi, rightyind, 0) & tm;
        if (tx != ty)
          break;
      }
    }
    if (rev)
    {
      if (--p < leftx)
        return -1;
      if (--xpos < 0)
      {
        xpos = B_SHIFT - 1;
        x = xs[--xind];
        nextx = (xind <= leftxind) ? 0 : xs[xind-1];
      }
      else
      {
        x <<= 1;
        if (nextx & MAXBIT)
          x |= 1;
        nextx <<= 1;
      }
    }
    else
    {
      if (++p > rightx)
        return -1;
      if (++xpos == B_SHIFT)
      {
        xpos = 0;
        x = xs[++xind];
        nextx = (xind >= rightxind) ? 0 : xs[xind+1];
      }
      else
      {
        x >>= 1;
        if (nextx & 1)
          x |= MAXBIT;
        else
          x &= ~MAXBIT;
        nextx >>= 1;
      }
    }
  }
}

int BitString::match(int startx, int lengthx, int exact, 
                     unsigned long* ys, int starty, int yl)
{
  unsigned long* xs = rep->s;
  int ylen = yl - starty;
  int righty = yl - 1;

  int rightx;
  int rev = startx < 0;
  if (rev)
  {
    rightx = lengthx + startx;
    startx = rightx - ylen + 1;
    if (exact && startx != 0)
      return 0;
  }
  else
  {
    rightx = lengthx - 1;
    if (exact && rightx - startx != righty)
      return 0;
  }

  if (righty < 0 || startx < 0 || startx >= lengthx)
    return 0;
  
  int xi   = startx / B_SHIFT;
  int xpos = startx % B_SHIFT;
  int yi   = starty / B_SHIFT;
  int ypos = starty % B_SHIFT;

  int rightxind = rightx / B_SHIFT;
  int rightyind = righty / B_SHIFT;
  int rightypos = righty % B_SHIFT;

  for (;;)
  {
    unsigned long x = borrow_hi(xs, xi, rightxind, xpos);
    unsigned long y = borrow_hi(ys, yi, rightyind, ypos);
    unsigned long m;
    if (yi == rightyind)
      m = rmask(rightypos);
    else if (yi+1 == rightyind)
      m = rmask(ypos + rightypos + 1);
    else
      m = ONES;

    if ((x & m) != y)
      return 0;
    else if (++yi > rightyind || ++xi > rightxind)
      return 1;
  }
}

int BitPattern::match(unsigned long* xs, int startx, int lengthx, int exact)
{
  unsigned long* ys = pattern.rep->s;
  int righty = pattern.rep->len - 1;
  unsigned long* ms = mask.rep->s;
  int rightm = mask.rep->len - 1;

  int rightx;
  int rev = startx < 0;
  if (rev)
  {
    rightx = lengthx + startx;
    startx = rightx - righty;
    if (exact && startx != 0)
      return 0;
  }
  else
  {
    rightx = lengthx - 1;
    if (exact && rightx - startx != righty)
      return 0;
  }

  if (righty < 0 || startx < 0 || startx >= lengthx)
    return 0;
  
  int xind = startx / B_SHIFT;
  int xpos = startx % B_SHIFT;
  int yind = 0;

  int rightxind = rightx / B_SHIFT;
  int rightyind = righty / B_SHIFT;
  int rightmind = rightm / B_SHIFT;

  for(;;)
  {
    unsigned long x = borrow_hi(xs, xind, rightxind, xpos);
    unsigned long y = borrow_hi(ys, yind, rightyind, 0);
    unsigned long m = borrow_hi(ms, yind, rightmind, 0);
    if ((x & m) != (y & m))
      return 0;
    else if (++yind > rightyind || ++xind > rightxind)
      return 1;
  }
}

BitSubString::BitSubString(BitString* x, int first, int l)
{
  if (first < 0 || l <= 0 || first + l > x->rep->len)
  {
    S = &_nil_BitString; pos = len = 0;
  }
  else
  {
    S = x; pos = first; len = l;
  }
}



void BitSubString::operator = (BitString& y)
{
  int ylen = y.rep->len;
  _BitStringrep* targ = S->rep;

  if (len == 0 || pos >= targ->len)
    return;

  int sl = targ->len - len + ylen;

  if (targ->ref != 1)
  {
    if (targ->ref > 0) targ->ref--;
    _BitStringrep* oldtarg = targ;
    targ = new_BitStringrep(sl);
    bit_transfer(oldtarg->s, 0, pos, targ->s, 0);
    bit_transfer(y.rep->s, 0, ylen, targ->s, pos);
    bit_transfer(oldtarg->s, pos+len, oldtarg->len, targ->s, pos + ylen);
  }
  else if (y.rep == targ || ylen > len)
  {
    _BitStringrep* oldtarg = targ;
    targ = new_BitStringrep(sl);
    bit_transfer(oldtarg->s, 0, pos, targ->s, 0);
    bit_transfer(y.rep->s, 0, ylen, targ->s, pos);
    bit_transfer(oldtarg->s, pos+len, oldtarg->len, targ->s, pos + ylen);
    delete oldtarg;
  }
  else if (len == ylen)
    bit_transfer(y.rep->s, 0, len, targ->s, pos);
  else if (ylen < len)
  {
    bit_transfer(y.rep->s, 0, ylen, targ->s, pos);
    bit_transfer(targ->s, pos+len, targ->len, targ->s, pos + ylen);
    targ->len = sl;
  }
  check_last(targ);
  S->rep = targ;
}

void BitSubString::operator = (BitSubString& y)
{
  _BitStringrep* targ = S->rep;

  if (len == 0 || pos >= targ->len)
    return;

  int sl = targ->len - len + y.len;

  if (targ->ref != 1)
  {
    if (targ->ref > 0) targ->ref--;
    _BitStringrep* oldtarg = targ;
    targ = new_BitStringrep(sl);
    bit_copy(oldtarg->s, targ->s, pos);
    bit_transfer(y.S->rep->s, y.pos, y.pos+y.len, targ->s, pos);
    bit_transfer(oldtarg->s, pos+len, oldtarg->len, targ->s, pos + y.len);
  }
  else if (y.S->rep == targ || y.len > len)
  {
    _BitStringrep* oldtarg = targ;
    targ = new_BitStringrep(sl);
    bit_copy(oldtarg->s, targ->s, pos);
    bit_transfer(y.S->rep->s, y.pos, y.pos+y.len, targ->s, pos);
    bit_transfer(oldtarg->s, pos+len, oldtarg->len, targ->s, pos + y.len);
    delete oldtarg;
  }
  else if (len == y.len)
    bit_transfer(y.S->rep->s, y.pos, y.pos+y.len, targ->s, pos);
  else if (y.len < len)
  {
    bit_transfer(y.S->rep->s, y.pos, y.pos+y.len, targ->s, pos);
    bit_transfer(targ->s, pos+len, targ->len, targ->s, pos + y.len);
    targ->len = sl;
  }
  check_last(targ);
  S->rep = targ;
}

BitSubString BitString::at(int first, int len)
{
  return BitSubString(this, first, len);
}

BitSubString BitString::before(int pos)
{
  return BitSubString(this, 0, pos);
}

BitSubString BitString::after(int pos)
{
  return BitSubString(this, pos + 1, rep->len - (pos + 1));
}

BitSubString BitString::at(BitString& y, int startpos = 0)
{
  int first = search(startpos, rep->len, y.rep->s, 0, y.rep->len);
  return BitSubString(this, first,  y.rep->len);
}

BitSubString BitString::before(BitString& y, int startpos = 0)
{
  int last = search(startpos, rep->len, y.rep->s, 0, y.rep->len);
  return BitSubString(this, 0, last);
}

BitSubString BitString::after(BitString& y, int startpos = 0)
{
  int first = search(startpos, rep->len, y.rep->s, 0, y.rep->len);
  return BitSubString(this, first + y.rep->len, rep->len - (first+y.rep->len));
}


BitSubString BitString::at(BitSubString& y, int startpos = 0)
{
  int first = search(startpos, rep->len, y.S->rep->s, y.pos, y.len);
  return BitSubString(this, first, y.len);
}

BitSubString BitString::before(BitSubString& y, int startpos = 0)
{
  int last = search(startpos, rep->len, y.S->rep->s, y.pos, y.len);
  return BitSubString(this, 0, last);
}

BitSubString BitString::after(BitSubString& y, int startpos = 0)
{
  int first = search(startpos, rep->len, y.S->rep->s, y.pos, y.len);
  return BitSubString(this, first + y.len, rep->len - (first + y.len));
}

BitSubString BitString::at(BitPattern& r, int startpos = 0)
{
  int first = r.search(rep->s, startpos, rep->len);
  return BitSubString(this, first, r.pattern.rep->len);
}


BitSubString BitString::before(BitPattern& r, int startpos = 0)
{
  int first = r.search(rep->s, startpos, rep->len);
  return BitSubString(this, 0, first);
}

BitSubString BitString::after(BitPattern& r, int startpos = 0)
{
  int first = r.search(rep->s, startpos, rep->len);
  int last;
  if (first < 0)
    last = -1;
  else
    last = first + r.pattern.rep->len;
  return BitSubString(this, last, rep->len - last);
}

BitString common_prefix(BitString& x, BitString& y, int startpos = 0)
{
  BitString r;
  int xl = x.rep->len;
  int yl = y.rep->len;

  int startx, starty;
  if (startpos < 0)
  {
    startx = xl + startpos;
    starty = yl + startpos;
  }
  else
    startx = starty = startpos;

  if (startx < 0 || startx >= xl || starty < 0 || starty >= yl)
    return r;

  unsigned long* xs = &(x.rep->s[startx / B_SHIFT]);
  unsigned long a = *xs++;
  int xp = startx;

  unsigned long* ys = &(y.rep->s[starty / B_SHIFT]);
  unsigned long b = *ys++;
  int yp = starty;

  for(; xp < xl && yp < yl; ++xp, ++yp)
  {
    unsigned long xbit = 1 << (xp % B_SHIFT);
    unsigned long ybit = 1 << (yp % B_SHIFT);
    if (((a & xbit) == 0) != ((b & ybit) == 0))
      break;
    if (xbit == MAXBIT)
      a = *xs++;
    if (ybit == MAXBIT)
      b = *ys++;
  }
  if (xp > startx)
    r.copy(x.rep->s, startx, xp);
  return r;
}


BitString common_suffix(BitString& x, BitString& y, int startpos = -1)
{
  BitString r;
  int xl = x.rep->len;
  int yl = y.rep->len;

  int startx, starty;
  if (startpos < 0)
  {
    startx = xl + startpos;
    starty = yl + startpos;
  }
  else
    startx = starty = startpos;

  if (startx < 0 || startx >= xl || starty < 0 || starty >= yl)
    return r;

  unsigned long* xs = &(x.rep->s[startx / B_SHIFT]);
  unsigned long a = *xs--;
  int xp = startx;

  unsigned long* ys = &(y.rep->s[starty / B_SHIFT]);
  unsigned long b = *ys--;
  int yp = starty;

  for(; xp >= 0 && yp >= 0; --xp, --yp)
  {
    unsigned long xbit = 1 << (xp % B_SHIFT);
    unsigned long ybit = 1 << (yp % B_SHIFT);
    if (((a & xbit) == 0) != ((b & ybit) == 0))
      break;
    if (xbit == 0)
      a = *xs--;
    if (ybit == 0)
      b = *ys--;
  }
  if (xp < startx)
    r.copy(x.rep->s, xp+1, startx+1);
  return r;
}

BitString reverse(BitString& x)
{
  BitString y = x;
  y.make_unique();
  int yl = y.rep->len;
  int l = 0;
  int r = yl - 1;
  unsigned long* ls = y.rep->s;
  unsigned long* rs = &(ls[nwords(yl) - 1]);
  unsigned long lm = 1 << (l % B_SHIFT);
  unsigned long rm = 1 << (r % B_SHIFT);
  for (; l < r; ++l, --r)
  {
    if (((*ls & lm) != 0) != ((*rs & rm) != 0))
    {
      *rs ^= rm;
      *ls ^= lm;
    }
    if (lm == MAXBIT)
    {
      ++ls;
      lm = 1;
    }
    else
      lm <<= 1;
    if (rm == 1)
    {
      --rs;
      rm = MAXBIT;
    }
    else
      rm = (rm >> 1) & ~(MAXBIT);
  }
  return y;
}
