#include "cupspageparser.h"

#include <qfile.h>
#include <qtextstream.h>
#include <qmap.h>
#include <qpopupmenu.h>
#include <qdatetimeedit.h>
#include <qlistview.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kdialogbase.h>

#include <stdlib.h>

CupsPageParser::CupsPageParser( QObject *parent )
	: CupsParser( "PageParser", "/var/log/cups/page_log", parent )
{
	m_re.setPattern( "(\\S+) (\\S+) (\\d+) \\[([^]]+)\\] (\\d+) (\\d+) (.*)" );
	m_widget = 0;
}

CupsPageParser::~CupsPageParser()
{
}

QStringList CupsPageParser::fields()
{
	return QStringList() << i18n( "Printer" ) << i18n( "User name" ) << i18n( "Job ID" )
		<< i18n( "Page number" ) << i18n( "Number of copies" ) << i18n( "Billing information" );
}

CupsParser::Item* CupsPageParser::lineToItem( const QString& line )
{
	Item *item = 0;
	if ( m_re.exactMatch( line ) )
	{
		item = new Item( 6 );
		item->fields[ 0 ] = m_re.cap( 1 );
		item->fields[ 1 ] = m_re.cap( 2 );
		item->fields[ 2 ] = m_re.cap( 3 ).toInt();
		item->fields[ 3 ] = m_re.cap( 5 ).toInt();
		item->fields[ 4 ] = m_re.cap( 6 ).toInt();
		item->fields[ 5 ] = m_re.cap( 7 );
		item->parseDate( m_re.cap( 4 ) );
	}
	return item;
}

void CupsPageParser::computePages( const QString& u, const QDateTime& since )
{
	QString user = ( u.isEmpty() ? QString( getenv( "USER" ) ) : u );
	QFile f( fileName() );
	if ( f.open( IO_ReadOnly ) )
	{
		QTextStream t( &f );
		QString line;
		CupsParser::Item *item = 0;

		while ( !t.atEnd() )
		{
			item = lineToItem( t.readLine() );
			if ( item && item->date.isValid() && item->date >= since )
				break;
			delete item;
			item = 0;
		}

		QMap<QString, int> map;
		while ( item && item->date.isValid() )
		{
			if ( item->fields[ 1 ].asString() == user )
			{
				QString prt = item->fields[ 0 ].asString();
				if ( map.contains( prt ) )
					map[ prt ] = map[ prt ] + item->fields[ 4 ].asInt();
				else
					map[ prt ] = item->fields[ 4 ].asInt();
			}
			if ( t.atEnd() )
				break;
			item = lineToItem( t.readLine() );
		}
		f.close();

		if ( map.count() > 0 )
		{
			QString txt = i18n( "Pages printed by %1 since %2" ).arg( user ).arg( since.toString() );
			txt.append( "<table>" );
			for ( QMap<QString, int>::ConstIterator it=map.begin(); it!= map.end(); ++it )
				txt.append( "<tr><td align=\"right\"><b>" + it.key() + ":</b></td><td>" + QString::number( *it ) + "</td></tr>" );
			txt.prepend( "<qt>" ).append( "</table></qt>" );
			KMessageBox::information( m_widget, txt );
		}
		else
			KMessageBox::information( m_widget, i18n( "No pages printed by %1 since %2." ).arg( user ).arg( since.toString() ) );
	}
	else
		KMessageBox::error( m_widget, i18n( "Unable to open file for reading: %1." ).arg( fileName() ) );
}

void CupsPageParser::computePagesYear()
{
	QDate d = QDate::currentDate();
	QDateTime dt( QDate( d.year(), 1, 1 ) );
	computePages( m_user, dt );
}

void CupsPageParser::computePagesMonth()
{
	QDate d = QDate::currentDate();
	QDateTime dt( QDate( d.year(), d.month(), 1 ) );
	computePages( m_user, dt );
}

void CupsPageParser::computePagesWeek()
{
	QDate d = QDate::currentDate();
	QDateTime dt( d.addDays( 1-d.dayOfWeek() ) );
	computePages( m_user, dt );
}

void CupsPageParser::computePagesDay()
{
	QDate d = QDate::currentDate();
	QDateTime dt( d );
	computePages( m_user, dt );
}

void CupsPageParser::computePagesOther()
{
	KDialogBase dlg( m_widget, 0, true, i18n( "Select date" ), KDialogBase::Ok|KDialogBase::Cancel );
	QDateTimeEdit *dt = new QDateTimeEdit( &dlg );
	dt->setDateTime( QDateTime::currentDateTime() );
	dlg.setMainWidget( dt );
	if ( dlg.exec() )
		computePages( m_user, dt->dateTime() );
}

void CupsPageParser::setupRMB( QListViewItem *item, QPopupMenu *menu )
{
	if ( item )
		m_user = item->text( 2 );
	else
		m_user = getenv( "USER" );
	m_widget = menu->parentWidget();

	QPopupMenu *submenu = new QPopupMenu;
	submenu->insertItem( i18n( "today" ), this, SLOT( computePagesDay() ) );
	submenu->insertItem( i18n( "this week" ), this, SLOT( computePagesWeek() ) );
	submenu->insertItem( i18n( "this month" ), this, SLOT( computePagesMonth() ) );
	submenu->insertItem( i18n( "this year" ), this, SLOT( computePagesYear() ) );
	submenu->insertSeparator();
	submenu->insertItem( i18n( "since..." ), this, SLOT( computePagesOther() ) );

	menu->insertSeparator();
	menu->insertItem( i18n( "Pages printed by %1" ).arg( m_user ), submenu );
}

#include "cupspageparser.moc"
