<!--
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements.  See the NOTICE file
distributed with this work for additional information
regarding copyright ownership.  The ASF licenses this file
to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.
-->

## Scala

[Scala](https://www.scala-lang.org/) is an alternative language for the JVM. It doesn't support
annotation processing natively, so using it with the mapper is a bit more complicated, but it can be
done.

We have a full example at [DataStax-Examples/object-mapper-jvm/scala].

### Writing the model

You can use Scala case classes for your entities. Notice the peculiar syntax for field annotations:

```scala
@Entity
case class UserVideo(@(PartitionKey@field) userid: UUID,
                     @(ClusteringColumn@field)(0) addedDate: Instant,
                     @(ClusteringColumn@field)(1) videoid: UUID,
                     name: String,
                     previewImageLocation: String)
```

Case classes are immutable and use the [fluent getter style](../../entities#getter-style), but you
don't need to declare that explicitly with [@PropertyStrategy]: the mapper detects when it's
processing a case class, and will assume `mutable = false, getterStyle = FLUENT` by default.

The DAOs and main mapper can be defined as Scala traits, that are direct translations of their Java
equivalents:

```scala
@Dao
trait UserDao {
  @Select
  def get(userid: UUID): User
}
```

### Building

Since Scala does not support annotation processing, the mapper processor cannot operate on Scala
sources directly. But it can process the compiled class files output by the Scala compiler. So the
compilation happens in 3 phases:

1. Compile the Scala sources with the regular sbt task.
2. Execute a custom task that runs the annotation processor (`javac -proc:only ...`) on the compiled
   class files.
3. Execute another custom task that compiles the Java sources generated by the mapper.

See the example's [build.sbt] for the full details.

Because of that process, the sources fed to the processor cannot reference any generated code. So
the application code needs to be placed in a separate subproject, in order to have access to the
mapper builder.

[DataStax-Examples/object-mapper-jvm/scala]: https://github.com/DataStax-Examples/object-mapper-jvm/tree/master/scala
[build.sbt]: https://github.com/DataStax-Examples/object-mapper-jvm/blob/master/scala/build.sbt

[@PropertyStrategy]: https://docs.datastax.com/en/drivers/java/4.17/com/datastax/oss/driver/api/mapper/annotations/PropertyStrategy.html
