/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import bk-shade.com.google.common.annotations.VisibleForTesting;
import bk-shade.com.google.common.base.Charsets;
import bk-shade.com.google.common.collect.Lists;
import bk-shade.com.google.common.collect.Sets;
import bk-shade.com.google.common.util.concurrent.SettableFuture;
import bk-shade.com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Observable;
import java.util.Observer;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.bookkeeper.bookie.BookieCriticalThread;
import org.apache.bookkeeper.bookie.BookieException;
import org.apache.bookkeeper.bookie.BookieThread;
import org.apache.bookkeeper.bookie.CheckpointSourceList;
import org.apache.bookkeeper.bookie.Cookie;
import org.apache.bookkeeper.bookie.HandleFactory;
import org.apache.bookkeeper.bookie.HandleFactoryImpl;
import org.apache.bookkeeper.bookie.Journal;
import org.apache.bookkeeper.bookie.LedgerDescriptor;
import org.apache.bookkeeper.bookie.LedgerDirsManager;
import org.apache.bookkeeper.bookie.LedgerDirsMonitor;
import org.apache.bookkeeper.bookie.LedgerStorage;
import org.apache.bookkeeper.bookie.LedgerStorageFactory;
import org.apache.bookkeeper.bookie.SyncThread;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.meta.LedgerManager;
import org.apache.bookkeeper.meta.LedgerManagerFactory;
import org.apache.bookkeeper.net.BookieSocketAddress;
import org.apache.bookkeeper.net.DNS;
import org.apache.bookkeeper.proto.BookkeeperInternalCallbacks;
import org.apache.bookkeeper.stats.Counter;
import org.apache.bookkeeper.stats.Gauge;
import org.apache.bookkeeper.stats.NullStatsLogger;
import org.apache.bookkeeper.stats.OpStatsLogger;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.util.DiskChecker;
import org.apache.bookkeeper.util.IOUtils;
import org.apache.bookkeeper.util.MathUtils;
import org.apache.bookkeeper.util.ZkUtils;
import org.apache.bookkeeper.util.collections.ConcurrentLongHashMap;
import org.apache.bookkeeper.versioning.Version;
import org.apache.bookkeeper.versioning.Versioned;
import org.apache.bookkeeper.zookeeper.BoundExponentialBackoffRetryPolicy;
import org.apache.bookkeeper.zookeeper.ZooKeeperClient;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.mutable.MutableBoolean;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.ACL;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Bookie
extends BookieCriticalThread {
    private static final Logger LOG = LoggerFactory.getLogger(Bookie.class);
    final List<File> journalDirectories;
    final ServerConfiguration conf;
    final SyncThread syncThread;
    final LedgerManagerFactory ledgerManagerFactory;
    final LedgerManager ledgerManager;
    final LedgerStorage ledgerStorage;
    final List<Journal> journals;
    final HandleFactory handles;
    static final long METAENTRY_ID_LEDGER_KEY = -4096L;
    static final long METAENTRY_ID_FENCE_KEY = -8192L;
    protected final String bookieRegistrationPath;
    protected final String bookieReadonlyRegistrationPath;
    private final LedgerDirsManager ledgerDirsManager;
    private LedgerDirsManager indexDirsManager;
    LedgerDirsMonitor ledgerMonitor;
    LedgerDirsMonitor idxMonitor;
    ZooKeeper zk;
    private volatile boolean running = false;
    private volatile boolean shuttingdown = false;
    private int exitCode = 0;
    private final ConcurrentLongHashMap<byte[]> masterKeyCache = new ConcurrentLongHashMap();
    protected final String zkBookieRegPath;
    protected final String zkBookieReadOnlyPath;
    protected final List<ACL> zkAcls;
    private final AtomicBoolean zkRegistered = new AtomicBoolean(false);
    protected final AtomicBoolean readOnly = new AtomicBoolean(false);
    final ExecutorService stateService = Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setNameFormat("BookieStateService-%d").build());
    private final StatsLogger statsLogger;
    private final Counter writeBytes;
    private final Counter readBytes;
    private final OpStatsLogger addEntryStats;
    private final OpStatsLogger recoveryAddEntryStats;
    private final OpStatsLogger readEntryStats;
    private final OpStatsLogger addBytesStats;
    private final OpStatsLogger readBytesStats;
    AtomicBoolean shutdownTriggered = new AtomicBoolean(false);

    public static void checkDirectoryStructure(File dir) throws IOException {
        if (!dir.exists()) {
            File parent = dir.getParentFile();
            File preV3versionFile = new File(dir.getParent(), "VERSION");
            final AtomicBoolean oldDataExists = new AtomicBoolean(false);
            parent.list(new FilenameFilter(){

                @Override
                public boolean accept(File dir, String name) {
                    if (name.endsWith(".txn") || name.endsWith(".idx") || name.endsWith(".log")) {
                        oldDataExists.set(true);
                    }
                    return true;
                }
            });
            if (preV3versionFile.exists() || oldDataExists.get()) {
                String err = "Directory layout version is less than 3, upgrade needed";
                LOG.error(err);
                throw new IOException(err);
            }
            if (!dir.mkdirs()) {
                String err = "Unable to create directory " + dir;
                LOG.error(err);
                throw new IOException(err);
            }
        }
    }

    private void checkEnvironment(ZooKeeper zk) throws BookieException, IOException {
        ArrayList<File> allLedgerDirs = new ArrayList<File>(this.ledgerDirsManager.getAllLedgerDirs().size() + this.indexDirsManager.getAllLedgerDirs().size());
        allLedgerDirs.addAll(this.ledgerDirsManager.getAllLedgerDirs());
        if (this.indexDirsManager != this.ledgerDirsManager) {
            allLedgerDirs.addAll(this.indexDirsManager.getAllLedgerDirs());
        }
        if (zk == null) {
            for (File journalDirectory : this.journalDirectories) {
                Bookie.checkDirectoryStructure(journalDirectory);
            }
            for (File dir : allLedgerDirs) {
                Bookie.checkDirectoryStructure(dir);
            }
            return;
        }
        if (this.conf.getAllowStorageExpansion()) {
            Bookie.checkEnvironmentWithStorageExpansion(this.conf, zk, this.journalDirectories, allLedgerDirs);
            return;
        }
        try {
            boolean newEnv = false;
            ArrayList<File> missedCookieDirs = new ArrayList<File>();
            ArrayList<Cookie> journalCookies = Lists.newArrayList();
            for (File journalDirectory : this.journalDirectories) {
                try {
                    Cookie journalCookie = Cookie.readFromDirectory(journalDirectory);
                    journalCookies.add(journalCookie);
                    if (journalCookie.isBookieHostCreatedFromIp()) {
                        this.conf.setUseHostNameAsBookieID(false);
                        continue;
                    }
                    this.conf.setUseHostNameAsBookieID(true);
                }
                catch (FileNotFoundException fnf) {
                    newEnv = true;
                    missedCookieDirs.add(journalDirectory);
                }
            }
            String instanceId = Bookie.getInstanceId(this.conf, zk);
            Cookie.Builder builder = Cookie.generateCookie(this.conf);
            if (null != instanceId) {
                builder.setInstanceId(instanceId);
            }
            Cookie masterCookie = builder.build();
            Versioned<Cookie> zkCookie = null;
            try {
                zkCookie = Cookie.readFromZooKeeper(zk, this.conf);
                masterCookie.verifyIsSuperSet(zkCookie.getValue());
            }
            catch (KeeperException.NoNodeException noNodeException) {
                // empty catch block
            }
            for (File file : this.journalDirectories) {
                Bookie.checkDirectoryStructure(file);
            }
            if (!newEnv) {
                for (Cookie cookie : journalCookies) {
                    masterCookie.verify(cookie);
                }
            }
            for (File file : allLedgerDirs) {
                Bookie.checkDirectoryStructure(file);
                try {
                    Cookie c = Cookie.readFromDirectory(file);
                    masterCookie.verify(c);
                }
                catch (FileNotFoundException fnf) {
                    missedCookieDirs.add(file);
                }
            }
            if (!newEnv && missedCookieDirs.size() > 0) {
                Iterator existingLedgerDirs = Sets.newHashSet();
                for (Cookie journalCookie : journalCookies) {
                    Collections.addAll(existingLedgerDirs, journalCookie.getLedgerDirPathsFromCookie());
                }
                ArrayList<File> arrayList = new ArrayList<File>();
                ArrayList<File> nonEmptyDirs = new ArrayList<File>();
                for (File file : missedCookieDirs) {
                    if (existingLedgerDirs.contains(file.getParent())) {
                        arrayList.add(file);
                        continue;
                    }
                    String[] content = file.list();
                    if (content == null || content.length == 0) continue;
                    nonEmptyDirs.add(file);
                }
                if (arrayList.size() > 0 || nonEmptyDirs.size() > 0) {
                    LOG.error("Either not all local directories have cookies or directories being added  newly are not empty. Directories missing cookie file are: " + arrayList + " New directories that are not empty are: " + nonEmptyDirs);
                    throw new BookieException.InvalidCookieException();
                }
            }
            if (missedCookieDirs.size() > 0) {
                LOG.info("Stamping new cookies on all dirs {}", missedCookieDirs);
                for (File file : this.journalDirectories) {
                    masterCookie.writeToDirectory(file);
                }
                for (File file : allLedgerDirs) {
                    masterCookie.writeToDirectory(file);
                }
                masterCookie.writeToZooKeeper(zk, this.conf, zkCookie != null ? zkCookie.getVersion() : Version.NEW);
            }
            List<File> ledgerDirs = this.ledgerDirsManager.getAllLedgerDirs();
            this.checkIfDirsOnSameDiskPartition(ledgerDirs);
            List<File> list = this.indexDirsManager.getAllLedgerDirs();
            this.checkIfDirsOnSameDiskPartition(list);
            this.checkIfDirsOnSameDiskPartition(this.journalDirectories);
        }
        catch (KeeperException ke) {
            LOG.error("Couldn't access cookie in zookeeper", (Throwable)ke);
            throw new BookieException.InvalidCookieException(ke);
        }
        catch (UnknownHostException uhe) {
            LOG.error("Couldn't check cookies, networking is broken", (Throwable)uhe);
            throw new BookieException.InvalidCookieException(uhe);
        }
        catch (IOException ioe) {
            LOG.error("Error accessing cookie on disks", (Throwable)ioe);
            throw new BookieException.InvalidCookieException(ioe);
        }
        catch (InterruptedException ie) {
            LOG.error("Thread interrupted while checking cookies, exiting", (Throwable)ie);
            throw new BookieException.InvalidCookieException(ie);
        }
    }

    private void checkIfDirsOnSameDiskPartition(List<File> dirs) throws BookieException.DiskPartitionDuplicationException {
        boolean allowDiskPartitionDuplication = this.conf.isAllowMultipleDirsUnderSameDiskPartition();
        MutableBoolean isDuplicationFoundAndNotAllowed = new MutableBoolean(false);
        HashMap<FileStore, List> fileStoreDirsMap = new HashMap<FileStore, List>();
        for (File dir : dirs) {
            FileStore fileStore2;
            try {
                fileStore2 = Files.getFileStore(dir.toPath());
            }
            catch (IOException e) {
                LOG.error("Got IOException while trying to FileStore of {}", (Object)dir);
                throw new BookieException.DiskPartitionDuplicationException(e);
            }
            if (fileStoreDirsMap.containsKey(fileStore2)) {
                ((List)fileStoreDirsMap.get(fileStore2)).add(dir);
                continue;
            }
            ArrayList<File> dirsList2 = new ArrayList<File>();
            dirsList2.add(dir);
            fileStoreDirsMap.put(fileStore2, dirsList2);
        }
        fileStoreDirsMap.forEach((fileStore, dirsList) -> {
            if (dirsList.size() > 1) {
                if (allowDiskPartitionDuplication) {
                    LOG.warn("Dirs: {} are in same DiskPartition/FileSystem: {}", dirsList, fileStore);
                } else {
                    LOG.error("Dirs: {} are in same DiskPartition/FileSystem: {}", dirsList, fileStore);
                    isDuplicationFoundAndNotAllowed.setValue(true);
                }
            }
        });
        if (isDuplicationFoundAndNotAllowed.getValue().booleanValue()) {
            throw new BookieException.DiskPartitionDuplicationException();
        }
    }

    public static void checkEnvironmentWithStorageExpansion(ServerConfiguration conf, ZooKeeper zk, List<File> journalDirectories, List<File> allLedgerDirs) throws BookieException, IOException {
        try {
            boolean newEnv = false;
            ArrayList<File> missedCookieDirs = new ArrayList<File>();
            ArrayList<Cookie> journalCookies = Lists.newArrayList();
            for (File journalDirectory : journalDirectories) {
                try {
                    Cookie journalCookie = Cookie.readFromDirectory(journalDirectory);
                    journalCookies.add(journalCookie);
                    if (journalCookie.isBookieHostCreatedFromIp()) {
                        conf.setUseHostNameAsBookieID(false);
                        continue;
                    }
                    conf.setUseHostNameAsBookieID(true);
                }
                catch (FileNotFoundException fnf) {
                    newEnv = true;
                    missedCookieDirs.add(journalDirectory);
                }
            }
            String instanceId = Bookie.getInstanceId(conf, zk);
            Cookie.Builder builder = Cookie.generateCookie(conf);
            if (null != instanceId) {
                builder.setInstanceId(instanceId);
            }
            Cookie masterCookie = builder.build();
            Versioned<Cookie> zkCookie = null;
            try {
                zkCookie = Cookie.readFromZooKeeper(zk, conf);
                masterCookie.verifyIsSuperSet(zkCookie.getValue());
            }
            catch (KeeperException.NoNodeException noNodeException) {
                // empty catch block
            }
            for (File file : journalDirectories) {
                Bookie.checkDirectoryStructure(file);
            }
            if (!newEnv) {
                for (Cookie cookie : journalCookies) {
                    masterCookie.verifyIsSuperSet(cookie);
                }
            }
            for (File file : allLedgerDirs) {
                Bookie.checkDirectoryStructure(file);
                try {
                    Cookie c = Cookie.readFromDirectory(file);
                    masterCookie.verifyIsSuperSet(c);
                }
                catch (FileNotFoundException fnf) {
                    missedCookieDirs.add(file);
                }
            }
            if (!newEnv && missedCookieDirs.size() > 0) {
                HashSet existingLedgerDirs = Sets.newHashSet();
                for (Cookie journalCookie : journalCookies) {
                    Collections.addAll(existingLedgerDirs, journalCookie.getLedgerDirPathsFromCookie());
                }
                ArrayList<File> arrayList = new ArrayList<File>();
                ArrayList<File> nonEmptyDirs = new ArrayList<File>();
                for (File file : missedCookieDirs) {
                    if (existingLedgerDirs.contains(file.getParent())) {
                        arrayList.add(file);
                        continue;
                    }
                    String[] content = file.list();
                    if (content == null || content.length == 0) continue;
                    nonEmptyDirs.add(file);
                }
                if (arrayList.size() > 0 || nonEmptyDirs.size() > 0) {
                    LOG.error("Either not all local directories have cookies or directories being added  newly are not empty. Directories missing cookie file are: " + arrayList + " New directories that are not empty are: " + nonEmptyDirs);
                    throw new BookieException.InvalidCookieException();
                }
            }
            if (missedCookieDirs.size() > 0) {
                LOG.info("Stamping new cookies on all dirs {}", missedCookieDirs);
                for (File file : journalDirectories) {
                    masterCookie.writeToDirectory(file);
                }
                for (File file : allLedgerDirs) {
                    masterCookie.writeToDirectory(file);
                }
                masterCookie.writeToZooKeeper(zk, conf, zkCookie != null ? zkCookie.getVersion() : Version.NEW);
            }
        }
        catch (KeeperException ke) {
            LOG.error("Couldn't access cookie in zookeeper", (Throwable)ke);
            throw new BookieException.InvalidCookieException(ke);
        }
        catch (UnknownHostException uhe) {
            LOG.error("Couldn't check cookies, networking is broken", (Throwable)uhe);
            throw new BookieException.InvalidCookieException(uhe);
        }
        catch (IOException ioe) {
            LOG.error("Error accessing cookie on disks", (Throwable)ioe);
            throw new BookieException.InvalidCookieException(ioe);
        }
        catch (InterruptedException ie) {
            LOG.error("Thread interrupted while checking cookies, exiting", (Throwable)ie);
            throw new BookieException.InvalidCookieException(ie);
        }
    }

    public static BookieSocketAddress getBookieAddress(ServerConfiguration conf) throws UnknownHostException {
        BookieSocketAddress addr;
        String hostName;
        InetSocketAddress inetAddr;
        if (conf.getAdvertisedAddress() != null && conf.getAdvertisedAddress().trim().length() > 0) {
            String hostAddress = conf.getAdvertisedAddress().trim();
            return new BookieSocketAddress(hostAddress, conf.getBookiePort());
        }
        String iface = conf.getListeningInterface();
        if (iface == null) {
            iface = "default";
        }
        if ((inetAddr = new InetSocketAddress(hostName = DNS.getDefaultHost(iface), conf.getBookiePort())).isUnresolved()) {
            throw new UnknownHostException("Unable to resolve default hostname: " + hostName + " for interface: " + iface);
        }
        String hostAddress = inetAddr.getAddress().getHostAddress();
        if (conf.getUseHostNameAsBookieID()) {
            hostAddress = inetAddr.getAddress().getCanonicalHostName();
        }
        if ((addr = new BookieSocketAddress(hostAddress, conf.getBookiePort())).getSocketAddress().getAddress().isLoopbackAddress() && !conf.getAllowLoopback()) {
            throw new UnknownHostException("Trying to listen on loopback address, " + addr + " but this is forbidden by default (see ServerConfiguration#getAllowLoopback())");
        }
        return addr;
    }

    private static String getInstanceId(ServerConfiguration conf, ZooKeeper zk) throws KeeperException, InterruptedException {
        String instanceId = null;
        if (zk.exists(conf.getZkLedgersRootPath(), null) == null) {
            LOG.error("BookKeeper metadata doesn't exist in zookeeper. Has the cluster been initialized? Try running bin/bookkeeper shell metaformat");
            throw new KeeperException.NoNodeException("BookKeeper metadata");
        }
        try {
            byte[] data = zk.getData(conf.getZkLedgersRootPath() + "/" + "INSTANCEID", false, null);
            instanceId = new String(data, Charsets.UTF_8);
        }
        catch (KeeperException.NoNodeException e) {
            LOG.info("INSTANCEID not exists in zookeeper. Not considering it for data verification");
        }
        return instanceId;
    }

    public LedgerDirsManager getLedgerDirsManager() {
        return this.ledgerDirsManager;
    }

    LedgerDirsManager getIndexDirsManager() {
        return this.indexDirsManager;
    }

    public long getTotalDiskSpace() throws IOException {
        return this.getLedgerDirsManager().getTotalDiskSpace(this.ledgerDirsManager.getAllLedgerDirs());
    }

    public long getTotalFreeSpace() throws IOException {
        return this.getLedgerDirsManager().getTotalFreeSpace(this.ledgerDirsManager.getAllLedgerDirs());
    }

    public static File getCurrentDirectory(File dir) {
        return new File(dir, "current");
    }

    public static File[] getCurrentDirectories(File[] dirs) {
        File[] currentDirs = new File[dirs.length];
        for (int i = 0; i < dirs.length; ++i) {
            currentDirs[i] = Bookie.getCurrentDirectory(dirs[i]);
        }
        return currentDirs;
    }

    public Bookie(ServerConfiguration conf) throws IOException, KeeperException, InterruptedException, BookieException {
        this(conf, (StatsLogger)NullStatsLogger.INSTANCE);
    }

    public Bookie(ServerConfiguration conf, StatsLogger statsLogger) throws IOException, KeeperException, InterruptedException, BookieException {
        super("Bookie-" + conf.getBookiePort());
        this.statsLogger = statsLogger;
        this.zkAcls = ZkUtils.getACLs(conf);
        this.bookieRegistrationPath = conf.getZkAvailableBookiesPath() + "/";
        this.bookieReadonlyRegistrationPath = this.bookieRegistrationPath + "readonly";
        this.conf = conf;
        this.journalDirectories = Lists.newArrayList();
        for (File journalDirectory : conf.getJournalDirs()) {
            this.journalDirectories.add(Bookie.getCurrentDirectory(journalDirectory));
        }
        DiskChecker diskChecker = new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold());
        this.ledgerDirsManager = new LedgerDirsManager(conf, conf.getLedgerDirs(), diskChecker, statsLogger.scope("ledger"));
        File[] idxDirs = conf.getIndexDirs();
        this.indexDirsManager = null == idxDirs ? this.ledgerDirsManager : new LedgerDirsManager(conf, idxDirs, diskChecker, statsLogger.scope("index"));
        this.zk = this.instantiateZookeeperClient(conf);
        this.checkEnvironment(this.zk);
        this.ledgerManagerFactory = LedgerManagerFactory.newLedgerManagerFactory(conf, this.zk);
        LOG.info("instantiate ledger manager {}", (Object)this.ledgerManagerFactory.getClass().getName());
        this.ledgerManager = this.ledgerManagerFactory.newLedgerManager();
        this.ledgerMonitor = new LedgerDirsMonitor(conf, diskChecker, this.ledgerDirsManager);
        try {
            this.ledgerMonitor.init();
        }
        catch (LedgerDirsManager.NoWritableLedgerDirException nle) {
            if (!conf.isReadOnlyModeEnabled()) {
                throw nle;
            }
            this.transitionToReadOnlyMode();
        }
        if (null == idxDirs) {
            this.idxMonitor = this.ledgerMonitor;
        } else {
            this.idxMonitor = new LedgerDirsMonitor(conf, diskChecker, this.indexDirsManager);
            try {
                this.idxMonitor.init();
            }
            catch (LedgerDirsManager.NoWritableLedgerDirException nle) {
                if (!conf.isReadOnlyModeEnabled()) {
                    throw nle;
                }
                this.transitionToReadOnlyMode();
            }
        }
        String myID = this.getMyId();
        this.zkBookieRegPath = this.bookieRegistrationPath + myID;
        this.zkBookieReadOnlyPath = this.bookieReadonlyRegistrationPath + "/" + myID;
        this.journals = Lists.newArrayList();
        for (int i = 0; i < this.journalDirectories.size(); ++i) {
            this.journals.add(new Journal(this.journalDirectories.get(i), conf, this.ledgerDirsManager, statsLogger.scope("journal_" + i)));
        }
        CheckpointSourceList checkpointSource = new CheckpointSourceList(this.journals);
        String ledgerStorageClass = conf.getLedgerStorageClass();
        LOG.info("Using ledger storage: {}", (Object)ledgerStorageClass);
        this.ledgerStorage = LedgerStorageFactory.createLedgerStorage(ledgerStorageClass);
        this.ledgerStorage.initialize(conf, this.ledgerManager, this.ledgerDirsManager, this.indexDirsManager, checkpointSource, statsLogger);
        this.syncThread = new SyncThread(conf, this.getLedgerDirsListener(), this.ledgerStorage, checkpointSource);
        this.handles = new HandleFactoryImpl(this.ledgerStorage);
        this.writeBytes = statsLogger.getCounter("WRITE_BYTES");
        this.readBytes = statsLogger.getCounter("READ_BYTES");
        this.addEntryStats = statsLogger.getOpStatsLogger("BOOKIE_ADD_ENTRY");
        this.recoveryAddEntryStats = statsLogger.getOpStatsLogger("BOOKIE_RECOVERY_ADD_ENTRY");
        this.readEntryStats = statsLogger.getOpStatsLogger("BOOKIE_READ_ENTRY");
        this.addBytesStats = statsLogger.getOpStatsLogger("BOOKIE_ADD_ENTRY_BYTES");
        this.readBytesStats = statsLogger.getOpStatsLogger("BOOKIE_READ_ENTRY_BYTES");
        statsLogger.registerGauge("SERVER_STATUS", (Gauge)new Gauge<Number>(){

            public Number getDefaultValue() {
                return 0;
            }

            public Number getSample() {
                return Bookie.this.zkRegistered.get() ? (Bookie.this.readOnly.get() ? 0 : 1) : -1;
            }
        });
    }

    private String getMyId() throws UnknownHostException {
        return Bookie.getBookieAddress(this.conf).toString();
    }

    void readJournal() throws IOException, BookieException {
        long startTs = MathUtils.now();
        Journal.JournalScanner scanner = new Journal.JournalScanner(){

            /*
             * Enabled force condition propagation
             * Lifted jumps to return sites
             */
            @Override
            public void process(int journalVersion, long offset, ByteBuffer recBuff) throws IOException {
                long ledgerId = recBuff.getLong();
                long entryId = recBuff.getLong();
                try {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Replay journal - ledger id : {}, entry id : {}.", (Object)ledgerId, (Object)entryId);
                    }
                    if (entryId == -4096L) {
                        if (journalVersion < 3) throw new IOException("Invalid journal. Contains journalKey  but layout version (" + journalVersion + ") is too old to hold this");
                        int masterKeyLen = recBuff.getInt();
                        byte[] masterKey = new byte[masterKeyLen];
                        recBuff.get(masterKey);
                        Bookie.this.masterKeyCache.put(ledgerId, masterKey);
                        return;
                    } else if (entryId == -8192L) {
                        if (journalVersion < 4) throw new IOException("Invalid journal. Contains fenceKey  but layout version (" + journalVersion + ") is too old to hold this");
                        byte[] key = (byte[])Bookie.this.masterKeyCache.get(ledgerId);
                        if (key == null) {
                            key = Bookie.this.ledgerStorage.readMasterKey(ledgerId);
                        }
                        LedgerDescriptor handle = Bookie.this.handles.getHandle(ledgerId, key);
                        handle.setFenced();
                        return;
                    } else {
                        byte[] key = (byte[])Bookie.this.masterKeyCache.get(ledgerId);
                        if (key == null) {
                            key = Bookie.this.ledgerStorage.readMasterKey(ledgerId);
                        }
                        LedgerDescriptor handle = Bookie.this.handles.getHandle(ledgerId, key);
                        recBuff.rewind();
                        handle.addEntry(Unpooled.wrappedBuffer((ByteBuffer)recBuff));
                    }
                    return;
                }
                catch (NoLedgerException nsle) {
                    if (!LOG.isDebugEnabled()) return;
                    LOG.debug("Skip replaying entries of ledger {} since it was deleted.", (Object)ledgerId);
                    return;
                }
                catch (BookieException be) {
                    throw new IOException(be);
                }
            }
        };
        for (Journal journal : this.journals) {
            journal.replay(scanner);
        }
        long elapsedTs = MathUtils.now() - startTs;
        LOG.info("Finished replaying journal in {} ms.", (Object)elapsedTs);
    }

    @Override
    public synchronized void start() {
        this.setDaemon(true);
        if (LOG.isDebugEnabled()) {
            LOG.debug("I'm starting a bookie with journal directories {}", (Object)this.journalDirectories.stream().map(File::getName).collect(Collectors.joining(", ")));
        }
        this.ledgerMonitor.start();
        if (this.indexDirsManager != this.ledgerDirsManager) {
            this.idxMonitor.start();
        }
        try {
            this.readJournal();
        }
        catch (IOException ioe) {
            LOG.error("Exception while replaying journals, shutting down", (Throwable)ioe);
            this.shutdown(5);
            return;
        }
        catch (BookieException be) {
            LOG.error("Exception while replaying journals, shutting down", (Throwable)be);
            this.shutdown(5);
            return;
        }
        LOG.info("Finished reading journal, starting bookie");
        super.start();
        this.ledgerDirsManager.addLedgerDirsListener(this.getLedgerDirsListener());
        if (this.indexDirsManager != this.ledgerDirsManager) {
            this.indexDirsManager.addLedgerDirsListener(this.getLedgerDirsListener());
        }
        this.ledgerStorage.start();
        this.syncThread.start();
        this.running = true;
        try {
            this.registerBookie(true).get();
        }
        catch (Exception e) {
            LOG.error("Couldn't register bookie with zookeeper, shutting down : ", (Throwable)e);
            this.shutdown(4);
        }
    }

    private LedgerDirsManager.LedgerDirsListener getLedgerDirsListener() {
        return new LedgerDirsManager.LedgerDirsListener(){

            @Override
            public void diskFull(File disk) {
            }

            @Override
            public void diskAlmostFull(File disk) {
            }

            @Override
            public void diskFailed(File disk) {
                Bookie.this.triggerBookieShutdown(5);
            }

            @Override
            public void allDisksFull() {
                Bookie.this.transitionToReadOnlyMode();
            }

            @Override
            public void fatalError() {
                LOG.error("Fatal error reported by ledgerDirsManager");
                Bookie.this.triggerBookieShutdown(5);
            }

            @Override
            public void diskWritable(File disk) {
                Bookie.this.transitionToWritableMode();
            }

            @Override
            public void diskJustWritable(File disk) {
                Bookie.this.transitionToWritableMode();
            }
        };
    }

    private ZooKeeper instantiateZookeeperClient(ServerConfiguration conf) throws IOException, InterruptedException, KeeperException {
        if (conf.getZkServers() == null) {
            LOG.warn("No ZK servers passed to Bookie constructor so BookKeeper clients won't know about this server!");
            return null;
        }
        return this.newZookeeper(conf);
    }

    protected boolean checkRegNodeAndWaitExpired(String regPath) throws IOException {
        final CountDownLatch prevNodeLatch = new CountDownLatch(1);
        Watcher zkPrevRegNodewatcher = new Watcher(){

            public void process(WatchedEvent event) {
                if (Watcher.Event.EventType.NodeDeleted == event.getType()) {
                    prevNodeLatch.countDown();
                }
            }
        };
        try {
            Stat stat = this.zk.exists(regPath, zkPrevRegNodewatcher);
            if (null != stat) {
                if (stat.getEphemeralOwner() != this.zk.getSessionId()) {
                    LOG.info("Previous bookie registration znode: {} exists, so waiting zk sessiontimeout: {} ms for znode deletion", (Object)regPath, (Object)this.conf.getZkTimeout());
                    if (!prevNodeLatch.await(this.conf.getZkTimeout(), TimeUnit.MILLISECONDS)) {
                        throw new KeeperException.NodeExistsException(regPath);
                    }
                    return false;
                }
                return true;
            }
            return false;
        }
        catch (KeeperException ke) {
            LOG.error("ZK exception checking and wait ephemeral znode {} expired : ", (Object)regPath, (Object)ke);
            throw new IOException("ZK exception checking and wait ephemeral znode " + regPath + " expired", ke);
        }
        catch (InterruptedException ie) {
            LOG.error("Interrupted checking and wait ephemeral znode {} expired : ", (Object)regPath, (Object)ie);
            throw new IOException("Interrupted checking and wait ephemeral znode " + regPath + " expired", ie);
        }
    }

    protected Future<Void> registerBookie(final boolean throwException) {
        return this.stateService.submit(new Callable<Void>(){

            @Override
            public Void call() throws IOException {
                try {
                    Bookie.this.doRegisterBookie();
                }
                catch (IOException ioe) {
                    if (throwException) {
                        throw ioe;
                    }
                    LOG.error("Couldn't register bookie with zookeeper, shutting down : ", (Throwable)ioe);
                    Bookie.this.triggerBookieShutdown(4);
                }
                return null;
            }
        });
    }

    protected void doRegisterBookie() throws IOException {
        this.doRegisterBookie(this.readOnly.get() ? this.zkBookieReadOnlyPath : this.zkBookieRegPath);
    }

    private void doRegisterBookie(String regPath) throws IOException {
        if (null == this.zk) {
            return;
        }
        this.zkRegistered.set(false);
        try {
            if (!this.checkRegNodeAndWaitExpired(regPath)) {
                this.zk.create(regPath, new byte[0], this.zkAcls, CreateMode.EPHEMERAL);
                LOG.info("Registered myself in ZooKeeper at {}.", (Object)regPath);
            }
            this.zkRegistered.set(true);
        }
        catch (KeeperException ke) {
            LOG.error("ZK exception registering ephemeral Znode for Bookie!", (Throwable)ke);
            throw new IOException(ke);
        }
        catch (InterruptedException ie) {
            LOG.error("Interrupted exception registering ephemeral Znode for Bookie!", (Throwable)ie);
            throw new IOException(ie);
        }
    }

    private Future<Void> transitionToWritableMode() {
        return this.stateService.submit(new Callable<Void>(){

            @Override
            public Void call() throws Exception {
                Bookie.this.doTransitionToWritableMode();
                return null;
            }
        });
    }

    @VisibleForTesting
    public void doTransitionToWritableMode() {
        if (this.shuttingdown) {
            return;
        }
        if (!this.readOnly.compareAndSet(true, false)) {
            return;
        }
        LOG.info("Transitioning Bookie to Writable mode and will serve read/write requests.");
        if (null == this.zk) {
            return;
        }
        try {
            this.doRegisterBookie(this.zkBookieRegPath);
        }
        catch (IOException e) {
            LOG.warn("Error in transitioning back to writable mode : ", (Throwable)e);
            this.transitionToReadOnlyMode();
            return;
        }
        try {
            this.zk.delete(this.zkBookieReadOnlyPath, -1);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOG.warn("Interrupted clearing readonly state while transitioning to writable mode : ", (Throwable)e);
            return;
        }
        catch (KeeperException e) {
            LOG.warn("Failed to delete bookie readonly state in zookeeper : ", (Throwable)e);
            return;
        }
    }

    private Future<Void> transitionToReadOnlyMode() {
        return this.stateService.submit(new Callable<Void>(){

            @Override
            public Void call() {
                Bookie.this.doTransitionToReadOnlyMode();
                return null;
            }
        });
    }

    @VisibleForTesting
    public void doTransitionToReadOnlyMode() {
        if (this.shuttingdown) {
            return;
        }
        if (!this.readOnly.compareAndSet(false, true)) {
            return;
        }
        if (!this.conf.isReadOnlyModeEnabled()) {
            LOG.warn("ReadOnly mode is not enabled. Can be enabled by configuring 'readOnlyModeEnabled=true' in configuration.Shutting down bookie");
            this.triggerBookieShutdown(5);
            return;
        }
        LOG.info("Transitioning Bookie to ReadOnly mode, and will serve only read requests from clients!");
        if (null == this.zk) {
            return;
        }
        try {
            if (null == this.zk.exists(this.bookieReadonlyRegistrationPath, false)) {
                try {
                    this.zk.create(this.bookieReadonlyRegistrationPath, new byte[0], this.zkAcls, CreateMode.PERSISTENT);
                }
                catch (KeeperException.NodeExistsException nodeExistsException) {
                    // empty catch block
                }
            }
            this.doRegisterBookie(this.zkBookieReadOnlyPath);
            try {
                this.zk.delete(this.zkBookieRegPath, -1);
            }
            catch (KeeperException.NoNodeException nne) {
                LOG.warn("No writable bookie registered node {} when transitioning to readonly", (Object)this.zkBookieRegPath, (Object)nne);
            }
        }
        catch (IOException e) {
            LOG.error("Error in transition to ReadOnly Mode. Shutting down", (Throwable)e);
            this.triggerBookieShutdown(5);
            return;
        }
        catch (KeeperException e) {
            LOG.error("Error in transition to ReadOnly Mode. Shutting down", (Throwable)e);
            this.triggerBookieShutdown(5);
            return;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOG.warn("Interrupted Exception while transitioning to ReadOnly Mode.");
            return;
        }
    }

    public boolean isReadOnly() {
        return this.readOnly.get();
    }

    private ZooKeeper newZookeeper(ServerConfiguration conf) throws IOException, InterruptedException, KeeperException {
        HashSet<Watcher> watchers = new HashSet<Watcher>();
        watchers.add(new Watcher(){

            public void process(WatchedEvent event) {
                if (!Bookie.this.running) {
                    return;
                }
                if (event.getType().equals((Object)Watcher.Event.EventType.None) && event.getState().equals((Object)Watcher.Event.KeeperState.Expired)) {
                    Bookie.this.zkRegistered.set(false);
                    Bookie.this.registerBookie(false);
                }
            }
        });
        return ZooKeeperClient.newBuilder().connectString(conf.getZkServers()).sessionTimeoutMs(conf.getZkTimeout()).watchers(watchers).operationRetryPolicy(new BoundExponentialBackoffRetryPolicy(conf.getZkRetryBackoffStartMs(), conf.getZkRetryBackoffMaxMs(), Integer.MAX_VALUE)).requestRateLimit(conf.getZkRequestRateLimit()).statsLogger(this.statsLogger.scope("bookie")).build();
    }

    public boolean isRunning() {
        return this.running;
    }

    @Override
    public void run() {
        try {
            for (Journal journal : this.journals) {
                journal.start();
            }
            for (Journal journal : this.journals) {
                journal.join();
            }
            LOG.info("Journal thread(s) quit.");
        }
        catch (InterruptedException ie) {
            LOG.warn("Interrupted on running journal thread : ", (Throwable)ie);
        }
        if (!this.shuttingdown) {
            LOG.error("Journal manager quits unexpectedly.");
            this.triggerBookieShutdown(5);
        }
    }

    void triggerBookieShutdown(final int exitCode) {
        if (!this.shutdownTriggered.compareAndSet(false, true)) {
            return;
        }
        LOG.info("Triggering shutdown of Bookie-{} with exitCode {}", (Object)this.conf.getBookiePort(), (Object)exitCode);
        BookieThread th = new BookieThread("BookieShutdownTrigger"){

            @Override
            public void run() {
                Bookie.this.shutdown(exitCode);
            }
        };
        th.start();
    }

    public int shutdown() {
        return this.shutdown(0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    synchronized int shutdown(int exitCode) {
        try {
            if (this.running) {
                LOG.info("Shutting down Bookie-{} with exitCode {}", (Object)this.conf.getBookiePort(), (Object)exitCode);
                if (this.exitCode == 0) {
                    this.exitCode = exitCode;
                }
                this.shuttingdown = true;
                LOG.info("Turning bookie to read only during shut down");
                this.readOnly.set(true);
                this.syncThread.shutdown();
                for (Journal journal : this.journals) {
                    journal.shutdown();
                }
                this.join();
                this.ledgerStorage.shutdown();
                try {
                    this.ledgerManager.close();
                    this.ledgerManagerFactory.uninitialize();
                }
                catch (IOException ie) {
                    LOG.error("Failed to close active ledger manager : ", (Throwable)ie);
                }
                this.ledgerMonitor.shutdown();
                if (this.indexDirsManager != this.ledgerDirsManager) {
                    this.idxMonitor.shutdown();
                }
                this.stateService.shutdown();
            }
            if (this.zk != null) {
                this.zk.close();
            }
        }
        catch (InterruptedException ie) {
            LOG.error("Interrupted during shutting down bookie : ", (Throwable)ie);
        }
        finally {
            this.running = false;
        }
        return this.exitCode;
    }

    private LedgerDescriptor getLedgerForEntry(ByteBuf entry, byte[] masterKey) throws IOException, BookieException {
        byte[] oldValue;
        long ledgerId = entry.getLong(entry.readerIndex());
        LedgerDescriptor l = this.handles.getHandle(ledgerId, masterKey);
        if (this.masterKeyCache.get(ledgerId) == null && (oldValue = this.masterKeyCache.putIfAbsent(ledgerId, masterKey)) == null) {
            ByteBuffer bb = ByteBuffer.allocate(20 + masterKey.length);
            bb.putLong(ledgerId);
            bb.putLong(-4096L);
            bb.putInt(masterKey.length);
            bb.put(masterKey);
            bb.flip();
            this.getJournal(ledgerId).logAddEntry(bb, (BookkeeperInternalCallbacks.WriteCallback)new NopWriteCallback(), null);
        }
        return l;
    }

    private Journal getJournal(long ledgerId) {
        return this.journals.get(MathUtils.signSafeMod(ledgerId, this.journals.size()));
    }

    private void addEntryInternal(LedgerDescriptor handle, ByteBuf entry, BookkeeperInternalCallbacks.WriteCallback cb, Object ctx) throws IOException, BookieException {
        long ledgerId = handle.getLedgerId();
        long entryId = handle.addEntry(entry);
        this.writeBytes.add((long)entry.readableBytes());
        if (LOG.isTraceEnabled()) {
            LOG.trace("Adding {}@{}", (Object)entryId, (Object)ledgerId);
        }
        this.getJournal(ledgerId).logAddEntry(entry, cb, ctx);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void recoveryAddEntry(ByteBuf entry, BookkeeperInternalCallbacks.WriteCallback cb, Object ctx, byte[] masterKey) throws IOException, BookieException {
        long requestNanos = MathUtils.nowInNano();
        boolean success = false;
        int entrySize = 0;
        try {
            LedgerDescriptor handle;
            LedgerDescriptor ledgerDescriptor = handle = this.getLedgerForEntry(entry, masterKey);
            synchronized (ledgerDescriptor) {
                entrySize = entry.readableBytes();
                this.addEntryInternal(handle, entry, cb, ctx);
            }
            success = true;
        }
        catch (LedgerDirsManager.NoWritableLedgerDirException e) {
            this.transitionToReadOnlyMode();
            throw new IOException(e);
        }
        finally {
            long elapsedNanos = MathUtils.elapsedNanos(requestNanos);
            if (success) {
                this.recoveryAddEntryStats.registerSuccessfulEvent(elapsedNanos, TimeUnit.NANOSECONDS);
                this.addBytesStats.registerSuccessfulValue((long)entrySize);
            } else {
                this.recoveryAddEntryStats.registerFailedEvent(elapsedNanos, TimeUnit.NANOSECONDS);
                this.addBytesStats.registerFailedValue((long)entrySize);
            }
            entry.release();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setExplicitLac(ByteBuf entry, Object ctx, byte[] masterKey) throws IOException, BookieException {
        try {
            LedgerDescriptor handle;
            long ledgerId = entry.getLong(entry.readerIndex());
            LedgerDescriptor ledgerDescriptor = handle = this.handles.getHandle(ledgerId, masterKey);
            synchronized (ledgerDescriptor) {
                handle.setExplicitLac(entry);
            }
        }
        catch (LedgerDirsManager.NoWritableLedgerDirException e) {
            this.transitionToReadOnlyMode();
            throw new IOException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ByteBuf getExplicitLac(long ledgerId) throws IOException, NoLedgerException {
        ByteBuf lac;
        LedgerDescriptor handle;
        LedgerDescriptor ledgerDescriptor = handle = this.handles.getReadOnlyHandle(ledgerId);
        synchronized (ledgerDescriptor) {
            lac = handle.getExplicitLac();
        }
        return lac;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addEntry(ByteBuf entry, BookkeeperInternalCallbacks.WriteCallback cb, Object ctx, byte[] masterKey) throws IOException, BookieException.LedgerFencedException, BookieException {
        long requestNanos = MathUtils.nowInNano();
        boolean success = false;
        int entrySize = 0;
        try {
            LedgerDescriptor handle;
            LedgerDescriptor ledgerDescriptor = handle = this.getLedgerForEntry(entry, masterKey);
            synchronized (ledgerDescriptor) {
                if (handle.isFenced()) {
                    throw BookieException.create(-101);
                }
                entrySize = entry.readableBytes();
                this.addEntryInternal(handle, entry, cb, ctx);
            }
            success = true;
        }
        catch (LedgerDirsManager.NoWritableLedgerDirException e) {
            this.transitionToReadOnlyMode();
            throw new IOException(e);
        }
        finally {
            long elapsedNanos = MathUtils.elapsedNanos(requestNanos);
            if (success) {
                this.addEntryStats.registerSuccessfulEvent(elapsedNanos, TimeUnit.NANOSECONDS);
                this.addBytesStats.registerSuccessfulValue((long)entrySize);
            } else {
                this.addEntryStats.registerFailedEvent(elapsedNanos, TimeUnit.NANOSECONDS);
                this.addBytesStats.registerFailedValue((long)entrySize);
            }
            entry.release();
        }
    }

    public SettableFuture<Boolean> fenceLedger(long ledgerId, byte[] masterKey) throws IOException, BookieException {
        LedgerDescriptor handle = this.handles.getHandle(ledgerId, masterKey);
        return handle.fenceAndLogInJournal(this.getJournal(ledgerId));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ByteBuf readEntry(long ledgerId, long entryId) throws IOException, NoLedgerException {
        long requestNanos = MathUtils.nowInNano();
        boolean success = false;
        int entrySize = 0;
        try {
            LedgerDescriptor handle = this.handles.getReadOnlyHandle(ledgerId);
            if (LOG.isTraceEnabled()) {
                LOG.trace("Reading {}@{}", (Object)entryId, (Object)ledgerId);
            }
            ByteBuf entry = handle.readEntry(entryId);
            this.readBytes.add((long)entry.readableBytes());
            success = true;
            ByteBuf byteBuf = entry;
            return byteBuf;
        }
        finally {
            long elapsedNanos = MathUtils.elapsedNanos(requestNanos);
            if (success) {
                this.readEntryStats.registerSuccessfulEvent(elapsedNanos, TimeUnit.NANOSECONDS);
                this.readBytesStats.registerSuccessfulValue((long)entrySize);
            } else {
                this.readEntryStats.registerFailedEvent(elapsedNanos, TimeUnit.NANOSECONDS);
                this.readBytesStats.registerFailedValue((long)entrySize);
            }
        }
    }

    public long readLastAddConfirmed(long ledgerId) throws IOException {
        LedgerDescriptor handle = this.handles.getReadOnlyHandle(ledgerId);
        return handle.getLastAddConfirmed();
    }

    public Observable waitForLastAddConfirmedUpdate(long ledgerId, long previoisLAC, Observer observer) throws IOException {
        LedgerDescriptor handle = this.handles.getReadOnlyHandle(ledgerId);
        return handle.waitForLastAddConfirmedUpdate(previoisLAC, observer);
    }

    public static boolean format(ServerConfiguration conf, boolean isInteractive, boolean force) {
        for (File journalDir : conf.getJournalDirs()) {
            File[] ledgerDirs;
            String[] journalDirFiles;
            String[] stringArray = journalDirFiles = journalDir.exists() && journalDir.isDirectory() ? journalDir.list() : null;
            if (journalDirFiles != null && journalDirFiles.length != 0) {
                try {
                    boolean confirm = false;
                    confirm = !isInteractive ? force : IOUtils.confirmPrompt("Are you sure to format Bookie data..?");
                    if (!confirm) {
                        LOG.error("Bookie format aborted!!");
                        return false;
                    }
                }
                catch (IOException e) {
                    LOG.error("Error during bookie format", (Throwable)e);
                    return false;
                }
            }
            if (!Bookie.cleanDir(journalDir)) {
                LOG.error("Formatting journal directory failed");
                return false;
            }
            for (File dir : ledgerDirs = conf.getLedgerDirs()) {
                if (Bookie.cleanDir(dir)) continue;
                LOG.error("Formatting ledger directory " + dir + " failed");
                return false;
            }
            File[] indexDirs = conf.getIndexDirs();
            if (null == indexDirs) continue;
            for (File dir : indexDirs) {
                if (Bookie.cleanDir(dir)) continue;
                LOG.error("Formatting ledger directory " + dir + " failed");
                return false;
            }
        }
        LOG.info("Bookie format completed successfully");
        return true;
    }

    private static boolean cleanDir(File dir) {
        if (dir.exists()) {
            File[] files = dir.listFiles();
            if (files != null) {
                for (File child : files) {
                    boolean delete = FileUtils.deleteQuietly((File)child);
                    if (delete) continue;
                    LOG.error("Not able to delete " + child);
                    return false;
                }
            }
        } else if (!dir.mkdirs()) {
            LOG.error("Not able to create the directory " + dir);
            return false;
        }
        return true;
    }

    public static void main(String[] args) throws IOException, InterruptedException, BookieException, KeeperException {
        Bookie b = new Bookie(new ServerConfiguration());
        b.start();
        CounterCallback cb = new CounterCallback();
        long start = MathUtils.now();
        for (int i = 0; i < 100000; ++i) {
            ByteBuf buff = Unpooled.buffer((int)1024);
            buff.writeLong(1L);
            buff.writeLong((long)i);
            cb.incCount();
            b.addEntry(buff, cb, null, new byte[0]);
        }
        cb.waitZero();
        long end = MathUtils.now();
        System.out.println("Took " + (end - start) + "ms");
    }

    public int getExitCode() {
        return this.exitCode;
    }

    static class CounterCallback
    implements BookkeeperInternalCallbacks.WriteCallback {
        int count;

        CounterCallback() {
        }

        @Override
        public synchronized void writeComplete(int rc, long l, long e, BookieSocketAddress addr, Object ctx) {
            --this.count;
            if (this.count == 0) {
                this.notifyAll();
            }
        }

        public synchronized void incCount() {
            ++this.count;
        }

        public synchronized void waitZero() throws InterruptedException {
            while (this.count > 0) {
                this.wait();
            }
        }
    }

    static class FutureWriteCallback
    implements BookkeeperInternalCallbacks.WriteCallback {
        SettableFuture<Boolean> result = SettableFuture.create();

        FutureWriteCallback() {
        }

        @Override
        public void writeComplete(int rc, long ledgerId, long entryId, BookieSocketAddress addr, Object ctx) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Finished writing entry {} @ ledger {} for {} : {}", new Object[]{entryId, ledgerId, addr, rc});
            }
            this.result.set(0 == rc);
        }

        public SettableFuture<Boolean> getResult() {
            return this.result;
        }
    }

    static class NopWriteCallback
    implements BookkeeperInternalCallbacks.WriteCallback {
        NopWriteCallback() {
        }

        @Override
        public void writeComplete(int rc, long ledgerId, long entryId, BookieSocketAddress addr, Object ctx) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Finished writing entry {} @ ledger {} for {} : {}", new Object[]{entryId, ledgerId, addr, rc});
            }
        }
    }

    public static class NoEntryException
    extends IOException {
        private static final long serialVersionUID = 1L;
        private final long ledgerId;
        private final long entryId;

        public NoEntryException(long ledgerId, long entryId) {
            this("Entry " + entryId + " not found in " + ledgerId, ledgerId, entryId);
        }

        public NoEntryException(String msg, long ledgerId, long entryId) {
            super(msg);
            this.ledgerId = ledgerId;
            this.entryId = entryId;
        }

        public long getLedger() {
            return this.ledgerId;
        }

        public long getEntry() {
            return this.entryId;
        }
    }

    public static class NoLedgerException
    extends IOException {
        private static final long serialVersionUID = 1L;
        private final long ledgerId;

        public NoLedgerException(long ledgerId) {
            super("Ledger " + ledgerId + " not found");
            this.ledgerId = ledgerId;
        }

        public long getLedgerId() {
            return this.ledgerId;
        }
    }
}

